/* High Level TDC V1190 Driver - MG xx/08/2005 @ LIXAM-CNRS*/
/* xx/02/2006 : Improvement of the structure tdc1190 for higher level programming */
/* xx/05/2006 : Add rtai and dma access */

/* --------------------------------------------------------------------------------------------------------------------- */
/* INCLUDES */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#ifdef ON_BOARD
  #include "bits.h"
  #include "/usr/include/math.h"
#else
  #include <math.h>
#endif
#include "tdc1190_driver_high.h"

/* --------------------------------------------------------------------------------------------------------------------- */
/* DEFINES */
#define MAX_NUM_CHANNEL         0x007F

#define ACQUSITION_MODE         0x0001
#define TRIG_WINDOW             0x0FFF
#define TRIG_OFFSET_S           0x0800
#define TRIG_OFFSET_V           0x07FF
#define TRIG_CONFIG             0x0001
#define EVENT_MAX_HITS          0x000F
#define EVENT_HEADER_TRAILER    0x0003
#define EDGE_CONFIGURATION      0x0003
#define EDGE_RESOLUTION_1       0x0003
#define EDGE_RESOLUTION_2       0x0007
#define WIDTH_RESOLUTION        0x0F00
#define CHANNEL_DEAD_TIME       0x0003
#define CHANNEL_OFFSET          0x00FF
#define HEADER_STATUS           0x0003
#define TDC_ERROR_STATUS        0x07FF
#define TDC_FIFO                0x0007
#define TDC_OFFSET_COARSE       0x07FF
#define TDC_OFFSET_FINE         0x001F

#define CANNOT_FIND_MAXHIT      TDC_ERROR|0x10
#define CANNOT_FIND_EDGE        TDC_ERROR|0x11
#define CANNOT_FIND_WIDTH       TDC_ERROR|0x12
#define CANNOT_FIND_DEADTIME    TDC_ERROR|0x13
#define CANNOT_FIND_FIFOSIZE    TDC_ERROR|0x14

/* --------------------------------------------------------------------------------------------------------------------- */
/* CONSTANT */
typedef char string[50];
static int MaxHit[11]={10,0,1,2,4,8,16,32,64,128,65536}; /* Nb hits */
static string Measurement[4]={"pair", "trailing edge", "leading edge","leading and trailing edges"};
static int Edge_Res[4]={3,800,200,100}; /* ps unit */
static int Pair_Res[9]={8,100,200,400,800,1600,3120,6250,12500}; /* ps unit */
static int Width_Res[15]={14,100,200,400,800,1600,3120,6250,12500,25000,50000,100000,200000,400000,800000}; /* ps unit */
static int Channel_DeadTime[5]={4,5,10,30,100}; /* ns unit */
static int FifoSize[9]={8,2,4,8,16,32,64,128,256}; /* 32-bit word unit */


/* --------------------------------------------------------------------------------------------------------------------- */
/* PROTOYPE */
static unsigned int Find_InTable(int *table, unsigned int value);
static int Get_SignedNumber(int value,int hsb,int lsb);
static int Set_SignedNumber(int value,int hsb,int lsb);

/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* READ FROM TDC MICRO CONTROLLER */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
int GetTdc_AcqMode(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_ACQ_MOD;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->Acquisition_Mode=(char)(op.operand[0]&ACQUSITION_MODE);
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_TriggerConfig(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_TRG_CONF;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;

  a_tdc->Trigger_Config[0]=(op.operand[0]&TRIG_WINDOW);                    /* window width - 12 bits*/
  a_tdc->Trigger_Config[0]=a_tdc->Trigger_Config[0]*25;

  a_tdc->Trigger_Config[1]=(op.operand[1]&TRIG_WINDOW);                    /* window offset- bit[11]-> sign; bit[0:10]->value*/
  a_tdc->Trigger_Config[1]=Get_SignedNumber(a_tdc->Trigger_Config[1],TRIG_OFFSET_S,TRIG_OFFSET_V)*25; 

  a_tdc->Trigger_Config[2]=(op.operand[2]&TRIG_WINDOW);                    /* margin extra search - 12 bits*/
  a_tdc->Trigger_Config[2]=a_tdc->Trigger_Config[2]*25;

  a_tdc->Trigger_Config[3]=(op.operand[3]&TRIG_WINDOW);                    /* margin reject - 12 bits*/
  a_tdc->Trigger_Config[3]=a_tdc->Trigger_Config[3]*25;

  a_tdc->Trigger_Config[4]=(op.operand[4]&TRIG_CONFIG);                    /* trigger substraction enable ?*/
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_MeasurementConfig(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_DETECTION;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->Measurement_Config=(op.operand[0]&EDGE_CONFIGURATION);
  sprintf(a_tdc->Measurement,"%s",Measurement[a_tdc->Measurement_Config]);
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_MeasurementResolution(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_RES;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  if (a_tdc->Measurement_Config){
    a_tdc->Edge_Resolution=Edge_Res[(op.operand[0]&EDGE_RESOLUTION_1)+1];
  }
  else {
    a_tdc->Edge_Resolution=Pair_Res[(op.operand[0]&EDGE_RESOLUTION_2)+1];
    a_tdc->Width_Resolution=Width_Res[((op.operand[0]&WIDTH_RESOLUTION)>>8)+1];
  }
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventMaxHit(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_EVENT_SIZE;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->Event_MaxHit=MaxHit[(op.operand[0]&EVENT_MAX_HITS)+1];
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventHeaderTrailer(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_HEAD_TRAILER;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->Event_HeaderTrailer=(char)(op.operand[0]&EVENT_HEADER_TRAILER);
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_ChannelsDeadTime(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_DEAD_TIME;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->Channel_DeadTime=Channel_DeadTime[(op.operand[0]&CHANNEL_DEAD_TIME)+1];
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_ChannelsActive(tdc1190 *a_tdc)
{ 
  int pattern[8];
  int bit,channel,numchannel;
  int res,i;
  opcode op;
  
  op.command=READ_EN_PATTERN;
  printf("ok\n");
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  printf("ok\n");
  for (i=0; i<op.nR;i++) pattern[i]=op.operand[i];
  for(numchannel=0;numchannel<a_tdc->NumberOfChannel;numchannel++) {
    channel=numchannel&0x0F;
    for (i=0,bit=1;i<channel;i++) bit*=2;
    a_tdc->Channel_Active[numchannel]=(pattern[(numchannel&0x70)>>4]&channel)>>channel;
    printf("numero pattern %i - numero channel %i <-> bx%s (%i)\n",(numchannel&0x70)>>4,
      channel, Set2Bit((unsigned int)bit),bit);
  }
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_ChannelsOffset(tdc1190 *a_tdc)
{
  int res,numchannel;
  opcode op;
  
  for(numchannel=0;numchannel<a_tdc->NumberOfChannel;numchannel++) {
    op.command=READ_ADJUST_CH;
    op.object=(char)(numchannel)&(a_tdc->NumberOfChannel);
    res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
    a_tdc->Channel_Offset[numchannel]=(op.operand[0]&CHANNEL_OFFSET);
  }
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_ErrorTypes(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_ERROR_TYPES;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->ErrorTypes=(op.operand[0]&TDC_ERROR_STATUS);
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_FifoSize(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_FIFO_SIZE;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->FifoSize=FifoSize[(op.operand[0]&TDC_FIFO)+1];
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_GlobalOffset(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=READ_GLOB_OFFS;
  res=read_micro(&a_tdc->tdc_access,&op); if (res) return res;
  a_tdc->TimeOffset[0]=(op.operand[0]& TDC_OFFSET_COARSE);
  a_tdc->TimeOffset[1]=(op.operand[1]& TDC_OFFSET_FINE);
  return NO_TDC_ERROR;
}

/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* WRITE TO TDC MICRO CONTROLLER */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
int SetTdc_AcqMode(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value -> Trigger Matching */
    op.command=TRG_MATCH;
  else
    op.command=CONT_STOR;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_AcqMode(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_KeepToken(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value -> Keep Token */
    op.command=SET_KEEP_TOKEN;
  else 
    op.command=CLEAR_KEEP_TOKEN;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_LoadConfig(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* default true value -> user */
    op.command=LOAD_USER_CONFIG;
  else
    op.command=LOAD_DEF_CONFIG;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  res=GetTdc_AcqMode(a_tdc); if (res) return res;
  res=GetTdc_TriggerConfig(a_tdc); if (res) return res;
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_SaveUserConfig(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=SAVE_USER_CONFIG;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_AutoLoadConfig(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* default true value -> user */
    op.command=AUTOLOAD_USER_CONF;
  else
    op.command=AUTOLOAD_DEF_CONF;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_Window(tdc1190 *a_tdc, unsigned int width, int offset)
{
  int res;
  opcode op;
  op.command=SET_WIN_WIDTH;
  op.operand[0]=((width/25)& TRIG_WINDOW); /* 25 ns base time*/
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  op.command=SET_WIN_OFFS;
  op.operand[0]=Set_SignedNumber(offset/25, TRIG_OFFSET_S, TRIG_OFFSET_V); /* 25 ns base time; sign may not work*/
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_TriggerConfig(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_Margin(tdc1190 *a_tdc, unsigned int extrasearch, unsigned int reject)
{
  int res;
  opcode op;
  op.command=SET_SW_MARGIN;
  op.operand[0]=((extrasearch/25)& TRIG_WINDOW); /* 25 ns base time*/
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  op.command=SET_REJ_MARGIN;
  op.operand[0]=((reject/25)& TRIG_WINDOW); /* 25 ns base time */
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;

  return GetTdc_TriggerConfig(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_TriggerSubstraction(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /*Default true value -> enable trigger time substraction */
    op.command=EN_SUB_TRG; 
  else
    op.command=DIS_SUB_TRG;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_TriggerConfig(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_MeasurementConfig(tdc1190 *a_tdc, unsigned int edge)
{
  int res;
  opcode op;
  op.command=SET_DETECTION;
  op.operand[0]=(edge&EDGE_CONFIGURATION);
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  res=GetTdc_MeasurementConfig(a_tdc);
  res+=GetTdc_MeasurementResolution(a_tdc);
  return res;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_MeasurementResolution(tdc1190 *a_tdc, unsigned int edge, unsigned int width)
{
  int res,dummy1, dummy2;
  opcode op;
  if (a_tdc->Measurement_Config)
    {
      dummy1=Find_InTable(Edge_Res,edge); if(dummy1==-1) return CANNOT_FIND_EDGE;
      op.command=SET_TR_LEAD_LSB;
      op.operand[0]=(dummy1&EDGE_RESOLUTION_1);
    }
  else
    {
      dummy2=Find_InTable(Pair_Res,edge);if(dummy2==-1) return CANNOT_FIND_EDGE;
      dummy1=Find_InTable(Width_Res,width);if(dummy1==-1) return CANNOT_FIND_WIDTH;
      op.command=SET_PAIR_RES;
      op.operand[0]=(((dummy1<<8)+dummy2)&(EDGE_RESOLUTION_2+WIDTH_RESOLUTION));
    }
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_MeasurementResolution(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_EventMaxHit(tdc1190 *a_tdc, int NbHits)
{
  int res,dummy;
  opcode op;
  op.command=SET_EVENT_SIZE;
  dummy=Find_InTable(MaxHit,NbHits);if(dummy==-1) return CANNOT_FIND_MAXHIT;
  op.operand[0]=dummy&EVENT_MAX_HITS;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_EventMaxHit(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_EventHeaderTrailer(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value -> enable header and trailer */
    op.command=EN_HEAD_TRAILER;
  else
    op.command=DIS_HEAD_TRAILER;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_EventHeaderTrailer(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ChannelsDeadTime(tdc1190 *a_tdc, int deadtime)
{
  int res;
  opcode op;
  op.command=SET_DEAD_TIME;
  deadtime=Find_InTable(Channel_DeadTime, deadtime); if (deadtime==-1) return CANNOT_FIND_DEADTIME;
  op.operand[0]=(deadtime&CHANNEL_DEAD_TIME);
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_ChannelsDeadTime(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ChannelsActive(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value ->  enable all channel*/
    op.command=EN_ALL_CH;
  else
    op.command=DIS_ALL_CH;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_ChannelsActive(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ChannelNActive(tdc1190 *a_tdc, int NumChannel, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value -> enable channel NumChannel */
    op.command=EN_CHANNEL;
  else
    op.command=DIS_CHANNEL;
  op.object=(char)(NumChannel)&(a_tdc->NumberOfChannel);
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_ChannelsActive(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ChannelNOffset(tdc1190 *a_tdc, int NumChannel, unsigned int offset)
{
  int res;
  opcode op;
  op.command=SET_ADJUST_CH;
  op.object=(char)(NumChannel)&(a_tdc->NumberOfChannel);
  op.operand[0]=(offset&CHANNEL_OFFSET);
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_ChannelsOffset(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ErrorMark(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value -> enable error mark */
    op.command=EN_ERROR_MARK;
  else
    op.command=DIS_ERROR_MARK;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ErrorBypass(tdc1190 *a_tdc, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value -> enable error mark */
    op.command=EN_ERROR_BYPASS;
  else
    op.command=DIS_ERROR_BYPASS;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return NO_TDC_ERROR;
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ErrorTypes(tdc1190 *a_tdc)
{
  int res;
  opcode op;
  op.command=SET_ERROR_TYPES;
  op.operand[0]= (a_tdc->ErrorTypes&TDC_ERROR_STATUS);
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_ErrorTypes(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_FifoSize(tdc1190 *a_tdc, int size)
{
  int res;
  opcode op;
  op.command=SET_FIFO_SIZE;
  size=Find_InTable(FifoSize,size);if (size==-1) return CANNOT_FIND_FIFOSIZE;
  op.operand[0]=size&TDC_FIFO;
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_FifoSize(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_GlobalOffset(tdc1190 *a_tdc, unsigned int coarse, unsigned int fine)
{
  int res;
  opcode op;
  op.command=SET_GLOB_OFFS;
  op.operand[0]=(coarse&TDC_OFFSET_COARSE);
  op.operand[1]=(fine&TDC_OFFSET_FINE);
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return GetTdc_GlobalOffset(a_tdc);
}

/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_TestMode(tdc1190 *a_tdc, unsigned int lsb_testword, unsigned int hsb_testword, int status)
{
  int res;
  opcode op;
  if (status) /* Default true value ->  enable all channel*/
    {
      op.command=ENABLE_TEST_MODE;
      op.operand[0]=lsb_testword;
      op.operand[1]=hsb_testword;
    }
  else
    op.command=DISABLE_TEST_MODE;
  
  res=write_micro(&a_tdc->tdc_access,&op); if (res) return res;
  return NO_TDC_ERROR;
}
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* READ FROM TDC REGISTERS */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
int GetTdc_Status(tdc1190 *a_tdc,unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,STATUS_REGISTER,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_Event(tdc1190 *a_tdc, unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,OUTPUT_REGISTER,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventCounter(tdc1190 *a_tdc, unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,EVENT_COUNTER,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventStored(tdc1190 *a_tdc, unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,EVENT_STORED,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_Firmware(tdc1190 *a_tdc,unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,FIRMWARE_VERSION,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventFIFO(tdc1190 *a_tdc,unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,EVENT_FIFO,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventFIFO_Stored(tdc1190 *a_tdc,unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,EVENT_FIFO_STORED,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_EventFIFO_Status(tdc1190 *a_tdc,unsigned int *value)
{
  int res;
  res=read_register(&a_tdc->tdc_access,EVENT_FIFO_STATUS,value);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_Control(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,CONTROL_REGISTER,&a_tdc->CtrlReg);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_InterruptLevel(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,INTERRUPT_LEVEL,&a_tdc->IrqLevel);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_InterruptVector(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,INTERRUPT_VECTOR,&a_tdc->IrqVector);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_GEO_Address(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,GEO_ADDRESS_REGISTER,&a_tdc->GeoAddress);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_MCST_Address(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,MCST_BASE_ADDRESS,&a_tdc->MCSTAddress);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_MCST_Control(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,MCST_CONTROL,&a_tdc->MCSTControl);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */

int GetTdc_AllmostFull(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,ALMOST_FULL_EVENT,&a_tdc->AlmostFull);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_BLT_EventNumber(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,BTL_EVENT_NUMBER,&a_tdc->BLTEventNumber);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_TestReg(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,TEST_REG,&a_tdc->TestReg);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_OutProg_Control(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,OUTPUT_PROG_CONTROL,&a_tdc->OutProg);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_Dummy32(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,DUMMY32,&a_tdc->Dummy32);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int GetTdc_Dummy16(tdc1190 *a_tdc)
{
  int res;
  res=read_register(&a_tdc->tdc_access,DUMMY16,&a_tdc->Dummy16);if (res) return res;
  return NO_TDC_ERROR;
}
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* WRITE TO TDC REGISTERS */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
int SetTdc_ModuleReset(tdc1190 *a_tdc)
{
  int res;
  res=write_register(&a_tdc->tdc_access,MODULE_RESET,0);if (res) return res;
  res=InitTdc(a_tdc,a_tdc->tdc_access.address_device,a_tdc->NumberOfChannel);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ClearRegister(tdc1190 *a_tdc)
{
  int res;
  res=write_register(&a_tdc->tdc_access,SOFTWARE_CLEAR,0);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_ClearEvent(tdc1190 *a_tdc)
{
  int res;
  res=write_register(&a_tdc->tdc_access,SOFTWARE_EVENT_RESET,0);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_SoftTrigger(tdc1190 *a_tdc)
{
  int res;
  res=write_register(&a_tdc->tdc_access,SOFTWARE_TRIGGER,0);if (res) return res;
  return NO_TDC_ERROR;
}
/* --------------------------------------------------------------------------------------------------------------------- */

int SetTdc_Control(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,CONTROL_REGISTER,value);if (res) return res;
  return GetTdc_Control(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */

int SetTdc_InterruptLevel(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,INTERRUPT_LEVEL,value);if (res) return res;
  return GetTdc_InterruptLevel(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_InterruptVector(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,INTERRUPT_VECTOR,value);if (res) return res;
  return GetTdc_InterruptVector(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_GEO_Address(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,GEO_ADDRESS_REGISTER,value);if (res) return res;
  return GetTdc_GEO_Address(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_MCST_Address(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,MCST_BASE_ADDRESS,value);if (res) return res;
  return GetTdc_MCST_Address(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_MCST_Control(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,MCST_CONTROL,value);if (res) return res;
  return GetTdc_MCST_Control(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */

int SetTdc_AlmostFull(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,ALMOST_FULL_EVENT,value);if (res) return res;
  return GetTdc_AllmostFull(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_BLT_EventNumber(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,BTL_EVENT_NUMBER,value);if (res) return res;
  return GetTdc_BLT_EventNumber(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_TestReg(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,TEST_REG,value);if (res) return res;
  return GetTdc_TestReg(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_OutProg_Control(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,OUTPUT_PROG_CONTROL,value);if (res) return res;
  return GetTdc_OutProg_Control(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_Dummy32(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,DUMMY32,value);if (res) return res;
  return GetTdc_Dummy32(a_tdc);
}
/* --------------------------------------------------------------------------------------------------------------------- */
int SetTdc_Dummy16(tdc1190 *a_tdc, unsigned int value)
{
  int res;
  res=write_register(&a_tdc->tdc_access,DUMMY16,value);if (res) return res;
  return GetTdc_Dummy16(a_tdc);
}

/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* RTAI AND DMA ACCESS */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */

/* --------------------------------------------------------------------------------------------------------------------- */
int startTdc_rtai(tdc1190 *a_tdc, unsigned int id, int size)
{ 
  char s[100];

  a_tdc->Tdc_Error=0;
  a_tdc->Hit_Error=0;
  a_tdc->totalEvent=0;
  a_tdc->totalTrig=0;
  a_tdc->totalAnode=0;
  a_tdc->totalTimer=0;
  a_tdc->RTAIoffset=0;

  sprintf(s,"/dev/rtf%i",1+2*id);
#ifdef ON_BOARD
  a_tdc->RTAIcounter_access = open("/dev/rtf1",O_RDWR);
  sprintf(s,"/dev/rtf%i",2+2*id);
  a_tdc->RTAIdata_access = open("/dev/rtf2",O_RDWR);
  if (a_tdc->RTAIcounter_access == -1 || a_tdc->RTAIdata_access == -1)
    {
      printf("Error in the real time fifo initialization !\n");
      return 1;
    }

  SetTdc_EventHeaderTrailer(a_tdc,0);  /*Set the event header to one in order to synchronize multiple tdc*/
  SetTdc_GEO_Address(a_tdc,id);
  SetTdc_AlmostFull(a_tdc,size); 
  SetTdc_InterruptLevel(a_tdc,6);	
 
  ioctl(a_tdc->tdc_access.fd,UNIVERSE_DEBUT,size);
  ioctl(a_tdc->tdc_access.fd,UNIVERSE_IRQ,BIT8);
  ioctl(a_tdc->tdc_access.fd,UNIVERSE_IRQ,BIT6);
#endif  
  return 0;	
}

/* --------------------------------------------------------------------------------------------------------------------- */
int readTdc_rtai(tdc1190 *a_tdc, int n)
{ 
  int i, count=0;
  unsigned long total_wc=0;
  

  if (a_tdc->RTAIoffset!=0) { /* a previous access has been done in manual mode*/ 
    if (n) { /*Manual mode -> just read  'n' 32bit words*/  
      /*printf("manu 2\n");*/         
      a_tdc->RTAIoffset-=n;
      if (a_tdc->RTAIoffset<0) { 
	n+=a_tdc->RTAIoffset;
	a_tdc->RTAIoffset=0;
      }
      total_wc=n;
    }
    else { /*Automatic mode -> empty buffer*/
      /*printf("auto 2\n");*/
      total_wc=a_tdc->RTAIoffset;
      a_tdc->RTAIoffset=0;
    }
  }
  else { /* no previous access has been done  in manual mode -> get a big buffer*/    
#ifdef ON_BOARD 

    count = read(a_tdc->RTAIcounter_access,a_tdc->RTAIcounter_fifo,sizeof(long)*WORDCOUNT)/4;
    for(i=0;i<count;i++) {
      switch (a_tdc->RTAIcounter_fifo[i]) {
      case -3 :
	break;
      case -2:
	break;
      case -1:
	break;
      default:
	total_wc += a_tdc->RTAIcounter_fifo[i];
	break;
      }
    }

#endif
    if (n) { /*Manual mode-> read 'n' 32bit words */
      a_tdc->RTAIoffset=total_wc-n;
      if (a_tdc->RTAIoffset<0) { 
	n+=a_tdc->RTAIoffset;
	a_tdc->RTAIoffset=0;
	/*printf("manu 1\n");*/
      }
      else {
	/*printf("auto 1\n");*/
      }
      total_wc=n;
    }
  }  
  a_tdc->RTAIcounter=total_wc;
  
#ifdef ON_BOARD 
  if (total_wc != 0) read(a_tdc->RTAIdata_access,a_tdc->RTAIdata_fifo,sizeof(unsigned long)*total_wc);
#endif 
  return a_tdc->RTAIcounter; 
}
  
/* --------------------------------------------------------------------------------------------------------------------- */
void stopTdc_rtai(tdc1190 *a_tdc)
{
#ifdef ON_BOARD
  /*Stop RTAI mode*/
  ioctl(a_tdc->tdc_access.fd,UNIVERSE_FIN,0);
  ioctl(a_tdc->tdc_access.fd,UNIVERSE_N_IRQ,BIT6);
  ioctl(a_tdc->tdc_access.fd,UNIVERSE_N_IRQ,BIT8);
  close(a_tdc->RTAIcounter_access);
  close(a_tdc->RTAIdata_access);
  status_rtai(a_tdc);
#endif
}
 
/* --------------------------------------------------------------------------------------------------------------------- */
void status_rtai(tdc1190 *a_tdc)
{
 printf("Total number of Trigger:%li, Total number of good events: %li\n", a_tdc->totalTrig, a_tdc->totalEvent);
  printf("Total number of Anode events: %li, Total number of Timer events: %li \n", a_tdc->totalAnode, a_tdc->totalTimer);
  printf("Number of hits too long=%li, Number of tdc Error=%li\n", a_tdc->Hit_Error,a_tdc->Tdc_Error);
}

/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* STRUCTURE ACCESS */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
int InitTdc(tdc1190 *a_tdc, unsigned int address, unsigned char NbChannel)
{
  int res;
  
  open_vme(&a_tdc->tdc_access,address);
  a_tdc->tdc_access.address_device=address;
  a_tdc->NumberOfChannel=NbChannel;
  a_tdc->tdc_access.fd=0;

  /* Registers*/
  res=GetTdc_Control(a_tdc); if (res) return res;
  res=GetTdc_InterruptLevel(a_tdc); if (res) return res;
  res=GetTdc_InterruptVector(a_tdc); if (res) return res;
  res=GetTdc_GEO_Address(a_tdc); if (res) return res;
  res=GetTdc_MCST_Address(a_tdc); if (res) return res;
  res=GetTdc_MCST_Control(a_tdc); if (res) return res;
  res=GetTdc_AllmostFull(a_tdc); if (res) return res;
  res=GetTdc_BLT_EventNumber(a_tdc); if (res) return res;
  res=GetTdc_TestReg(a_tdc); if (res) return res;
  res=GetTdc_OutProg_Control(a_tdc); if (res) return res;
  res=GetTdc_Dummy32(a_tdc); if (res) return res;
  res=GetTdc_Dummy16(a_tdc); if (res) return res;
  /* Micro Controler */
  res=GetTdc_AcqMode(a_tdc); if (res) return res;
  res=GetTdc_TriggerConfig(a_tdc);if (res) return res;
  res=GetTdc_MeasurementConfig(a_tdc);if (res) return res;
  res=GetTdc_MeasurementResolution(a_tdc);if (res) return res;
  res=GetTdc_EventMaxHit(a_tdc);if (res) return res;
  res=GetTdc_EventHeaderTrailer(a_tdc);if (res) return res;
  res=GetTdc_ChannelsDeadTime(a_tdc);if (res) return res;
  /*res=GetTdc_ChannelsActive(a_tdc);if (res) return res;*/
  res=GetTdc_ChannelsOffset(a_tdc);if (res) return res;
  res=GetTdc_ErrorTypes(a_tdc);if (res) return res;
  res=GetTdc_FifoSize(a_tdc);if (res) return res;
  res=GetTdc_GlobalOffset(a_tdc);if (res) return res;

 return NO_TDC_ERROR; 

}
/* --------------------------------------------------------------------------------------------------------------------- */
void PrintTdc_RegisterStatus(tdc1190 *a_tdc)
{
  unsigned int value;

  printf("========================== REGISTER STATUS =========================== \n");
  printf("Control Register :bx%s /  ",Set2Bit(a_tdc->CtrlReg));
  GetTdc_Status(a_tdc, &value);
  printf("Status Register :bx%s /  ",Set2Bit(value));
  printf("OUT_PROG Register:bx%s\n",Set2Bit(a_tdc->OutProg));  
  printf("Allmost full register Level :0x%04x  /  ",a_tdc->AlmostFull);
  printf("Irq Level:0x%04x  /  ",a_tdc->IrqLevel);
  printf("Irq Vector:0x%04x\n",a_tdc->IrqVector); 
  printf("GEO Address:0x%04x  /  ",a_tdc->GeoAddress);
  printf("MCST/CBLT Control:0x%04x  /  ",a_tdc->MCSTControl);  
  printf("MCST Base Address:0x%04x\n",a_tdc->MCSTAddress);
  printf("BLT Event Number:0x%04x\n",a_tdc->BLTEventNumber);
  printf("Test Register:0x%08x\n",a_tdc->TestReg);
  printf("Dummy 32:0x%08x  /  ",a_tdc->Dummy32);
  printf("Dummy 16:0x%04x\n\n",a_tdc->Dummy16);

}

/* --------------------------------------------------------------------------------------------------------------------- */
void PrintTdc_MicroStatus_1(tdc1190 *a_tdc)
{
  printf("=================== MICRO CONTROLLER STATUS I ======================== \n");
  if (a_tdc->Acquisition_Mode) printf("Acquisition Mode :Trigger Matching Mode\n");
  else printf("Acquisition Mode: Continuous Storage Mode\n");
  printf("Acquisition Window - Width: %i ns / ",a_tdc->Trigger_Config[0]);
  printf("Offset: %i ns  ",a_tdc->Trigger_Config[1]);
  printf("(0x%04x)\n",Set_SignedNumber(a_tdc->Trigger_Config[1]/25, TRIG_OFFSET_S, TRIG_OFFSET_V));
  printf("Trigger Margin - Extra search: %i ns / ",a_tdc->Trigger_Config[2]);
  printf("Reject: %i ns\n",a_tdc->Trigger_Config[3]);
  printf("-------------------------------------\n");
    if (a_tdc->Trigger_Config[4]) printf("Trigger substraction: enable\n");
  else  printf("Trigger substraction: disable\n");
  printf("Measurement: %s \n",a_tdc->Measurement);
  printf("Leading/Trailing edges resolution: %i ps\n",a_tdc->Edge_Resolution);
  if (!(a_tdc->Measurement_Config)) printf("Width resolution: %i ps\n",a_tdc->Width_Resolution);
  printf("-------------------------------------\n");
  printf("Event - Nb max of hits: %i / ",a_tdc->Event_MaxHit);
  printf("Channel Dead Time: %i (ns)\n",a_tdc->Channel_DeadTime);
}

/* --------------------------------------------------------------------------------------------------------------------- */
void PrintTdc_MicroStatus_2(tdc1190 *a_tdc)
{
  int i, j;
  char s[100];
  
  printf("=================== MICRO CONTROLLER STATUS II ======================== \n");
  if (a_tdc->Event_HeaderTrailer) printf("Header & Trailer: enable\n");
  else  printf("Header & Trailer: disable\n");

  printf("TDC Error Types : bx%s \n",Set2Bit(a_tdc->ErrorTypes));
  printf("TDC Fifo Size: %i words\n",a_tdc->FifoSize);
  printf("TDC Global Time Offset (coarse): %i  ",a_tdc->TimeOffset[0]);
  printf("(fine): %i\n\n",a_tdc->TimeOffset[1]);
  printf("-------------------------------------\n");
  printf("Channel status :\n");
  for (i=0;i<a_tdc->NumberOfChannel;i+=8) {
	s[0]=0;
	for (j=i;j<i+8;j++) {
		sprintf(s,"%s %i:%i-%i ",s,j,a_tdc->Channel_Active[j], a_tdc->Channel_Offset[j]);
		}
	 printf("%s\n",s);
	}
}
/* --------------------------------------------------------------------------------------------------------------------- */
void PrintTdc_Status(tdc1190 *a_tdc)
{
  PrintTdc_RegisterStatus(a_tdc);
  PrintTdc_MicroStatus_1(a_tdc);
  PrintTdc_MicroStatus_2(a_tdc);
}

/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* MISCELLANEOUS*/
/* ============================================================================================================================ */
/* ============================================================================================================================ */
/* ============================================================================================================================ */
static unsigned int Find_InTable(int *table, unsigned int value)
{
  int i=1,not_ok=1, max_value;

  max_value=*(table)+1;
  do {
    /*printf("%i <-> %i\n",*(table+i),value);*/
    if (*(table+i)==value) not_ok=0;
    i+=1;
  }
  while((not_ok)&&(i<max_value));
  if (i>max_value) return -1;
  else return i-2;
}
/* --------------------------------------------------------------------------------------------------------------------- */
static int Get_SignedNumber(int value,int hsb,int lsb)
{ 
  int sign=1;
  if (value&hsb) sign=-1;
  return ((value*sign)&lsb)*sign;
}
/* --------------------------------------------------------------------------------------------------------------------- */
static int Set_SignedNumber(int value,int hsb,int lsb)
{ 
  int sign=1;
  if (value<0) {
    value=-value;
    sign=-1;
  }
  return (value&lsb)*sign;
}
/* --------------------------------------------------------------------------------------------------------------------- */
char *Set2Bit(unsigned int value)
{ 
  char s1[33],s2[33];
  int i,hsb=0;
  s1[32]=0;
  for(i=0;i<32;i++){
    if ((value>>i)&1) {
      s1[31-i]='1';
      hsb=i;
    }
    else s1[31-i]='0';
  }
  for(i=0;i<=hsb+1;i++) {
    s2[i]=s1[31-hsb+i];
  }
  /*printf("bin2string : %s <-> %s, %f\n",s1,s2,atof(s2));*/
  return s2;
}
