/* ===========================================================
 * JFreeChart : a free chart library for the Java(tm) platform
 * ===========================================================
 *
 * (C) Copyright 2000-2004, by Object Refinery Limited and Contributors.
 *
 * Project Info:  http://www.jfree.org/jfreechart/index.html
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc. 
 * in the United States and other countries.]
 *
 * ------------------------
 * CandlestickRenderer.java
 * ------------------------
 * (C) Copyright 2001-2004, by Object Refinery Limited.
 *
 * Original Authors:  David Gilbert (for Object Refinery Limited);
 *                    Sylvain Vieujot;
 * Contributor(s):    Richard Atkinson;
 *                    Christian W. Zuckschwerdt;
 *                    Jerome Fisher;
 *
 * $Id: CandlestickRenderer.java,v 1.25 2004/01/05 17:11:47 mungady Exp $
 *
 * Changes
 * -------
 * 13-Dec-2001 : Version 1.  Based on code in the CandlestickPlot class, written by Sylvain
 *               Vieujot, which now is redundant (DG);
 * 23-Jan-2002 : Added DrawInfo parameter to drawItem(...) method (DG);
 * 28-Mar-2002 : Added a property change listener mechanism so that renderers no longer need to be
 *               immutable.  Added properties for up and down colors (DG);
 * 04-Apr-2002 : Updated with new automatic width calculation and optional volume display,
 *               contributed by Sylvain Vieujot (DG);
 * 09-Apr-2002 : Removed translatedRangeZero from the drawItem(...) method, and changed the return
 *               type of the drawItem method to void, reflecting a change in the XYItemRenderer
 *               interface.  Added tooltip code to drawItem(...) method (DG);
 * 25-Jun-2002 : Removed redundant code (DG);
 * 05-Aug-2002 : Small modification to drawItem method to support URLs for HTML image maps (RA);
 * 19-Sep-2002 : Fixed errors reported by Checkstyle (DG);
 * 25-Mar-2003 : Implemented Serializable (DG);
 * 01-May-2003 : Modified drawItem(...) method signature (DG);
 * 30-Jun-2003 : Added support for PlotOrientation (for completeness, this renderer is unlikely
 *               to be used with a HORIZONTAL orientation) (DG);
 * 30-Jul-2003 : Modified entity constructor (CZ);
 * 20-Aug-2003 : Implemented Cloneable and PublicCloneable (DG);
 * 29-Aug-2003 : Moved maxVolume calculation to initialise method (see bug report 796619) (DG);
 * 02-Sep-2003 : Added maxCandleWidthInMilliseconds as workaround for bug 796621 (DG);
 * 08-Sep-2003 : Changed ValueAxis API (DG);
 * 16-Sep-2003 : Changed ChartRenderingInfo --> PlotRenderingInfo (DG);
 * 13-Oct-2003 : Applied patch from Jerome Fisher to improve auto width calculations (DG);
 * 23-Dec-2003 : Fixed bug where up and down paint are used incorrectly (DG);
 * 
 */

package org.jfree.chart.renderer;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import org.jfree.chart.CrosshairInfo;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.entity.XYItemEntity;
import org.jfree.chart.labels.HighLowToolTipGenerator;
import org.jfree.chart.labels.XYToolTipGenerator;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.HighLowDataset;
import org.jfree.data.XYDataset;
import org.jfree.data.IntervalXYDataset;
import org.jfree.io.SerialUtilities;
import org.jfree.ui.RectangleEdge;
import org.jfree.util.PublicCloneable;

/**
 * A renderer that draws candlesticks on an {@link XYPlot} (requires a {@link HighLowDataset}).
 *
 * @author Sylvain Vieujot
 */
public class CandlestickRenderer extends AbstractXYItemRenderer 
                                 implements XYItemRenderer, 
                                            Cloneable,
                                            PublicCloneable,
                                            Serializable {
                                     
    /** The average width method. */                                          
    public static final int WIDTHMETHOD_AVERAGE = 0;
    
    /** The smallest width method. */
    public static final int WIDTHMETHOD_SMALLEST = 1;
    
    /** The interval data method. */
    public static final int WIDTHMETHOD_INTERVALDATA = 2;

    /** The method of automatically calculating the candle width. */
    private int autoWidthMethod = WIDTHMETHOD_AVERAGE;

    /** 
     * The number (generally between 0.0 and 1.0) by which the available space automatically 
     * calculated for the candles will be multiplied to determine the actual width to use. 
     */
    private double autoWidthFactor = 4.5 / 7;

    /** The minimum gap between one candle and the next */
    private double autoWidthGap = 0.0;

    /** The candle width. */
    private double candleWidth;
    
    /** The maximum candlewidth in milliseconds. */
    private double maxCandleWidthInMilliseconds = 1000.0 * 60.0 * 60.0 * 20.0;
    
    /** Temporary storage for the maximum candle width. */
    private double maxCandleWidth;

    /** The paint used to fill the candle when the price moved up from open to close. */
    private transient Paint upPaint;

    /** The paint used to fill the candle when the price moved down from open to close. */
    private transient Paint downPaint;

    /** A flag controlling whether or not volume bars are drawn on the chart. */
    private boolean drawVolume;
    
    /** Temporary storage for the maximum volume. */
    private transient double maxVolume;

    /**
     * Creates a new renderer for candlestick charts.
     */
    public CandlestickRenderer() {
        this(-1.0);
    }

    /**
     * Creates a new renderer for candlestick charts.
     * <P>
     * Use -1 for the candle width if you prefer the width to be calculated automatically.
     *
     * @param candleWidth  The candle width.
     */
    public CandlestickRenderer(double candleWidth) {
        this(candleWidth, true, new HighLowToolTipGenerator());
    }

    /**
     * Creates a new renderer for candlestick charts.
     * <P>
     * Use -1 for the candle width if you prefer the width to be calculated automatically.
     *
     * @param candleWidth  The candle width.
     * @param drawVolume  A flag indicating whether or not volume bars should be drawn.
     * @param toolTipGenerator  The tool tip generator. <code>null</code> is none.
     */
    public CandlestickRenderer(double candleWidth, boolean drawVolume,
                               XYToolTipGenerator toolTipGenerator) {

        super();
        setToolTipGenerator(toolTipGenerator);
        this.candleWidth = candleWidth;
        this.drawVolume = drawVolume;
        this.upPaint = Color.green;
        this.downPaint = Color.red;

    }

    /**
     * Returns the width of each candle.
     *
     * @return The candle width.
     * @see org.jfree.chart.renderer.CandlestickRenderer#setCandleWidth
     */
    public double getCandleWidth() {
        return this.candleWidth;
    }

    /**
     * Sets the candle width.
     * <P>
     * If you set the width to a negative value, the renderer will calculate
     * the candle width automatically based on the space available on the chart.
     *
     * @param width  The width.
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthMethod
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthGap
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthFactor
     * @see org.jfree.chart.renderer.CandlestickRenderer#setMaxCandleWidthInMilliseconds
     */
    public void setCandleWidth(double width) {

        if (width != this.candleWidth) {
            Double old = new Double(this.candleWidth);
            this.candleWidth = width;
            firePropertyChanged("CandleStickRenderer.candleWidth", old, new Double(width));
        }

    }

    /**
     * Returns the maximum width (in milliseconds) of each candle.
     *
     * @return The maximum candle width in milliseconds.
     */
    public double getMaxCandleWidthInMilliseconds() {
        return this.maxCandleWidthInMilliseconds;
    }

    /**
     * Sets the maximum candle width (in milliseconds).  
     *
     * @param millis  The maximum width.
     * @see org.jfree.chart.renderer.CandlestickRenderer#setCandleWidth
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthMethod
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthGap
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthFactor
     */
    public void setMaxCandleWidthInMilliseconds(double millis) {

        this.maxCandleWidthInMilliseconds = millis;
        firePropertyChanged("CandlestickRenderer.maxCandleWidthInMilliseconds", 
                            null, new Double(millis));

    }

    /**
     * Returns the method of automatically calculating the candle width.
     *
     * @return The method of automatically calculating the candle width.
     */
    public int getAutoWidthMethod() {
        return autoWidthMethod;
    }

    /**
     * Sets the method of automatically calculating the candle width.
     * <p>
     * <code>WIDTHMETHOD_AVERAGE</code>: Divides the entire display (ignoring scale factor) by the 
     * number of items, and uses this as the available width.<br>
     * <code>WIDTHMETHOD_SMALLEST</code>: Checks the interval between each item, and uses the 
     * smallest as the available width.<br>
     * <code>WIDTHMETHOD_INTERVALDATA</code>: Assumes that the dataset supports the 
     * IntervalXYDataset interface, and uses the startXValue - endXValue as the available width.
     * <br>
     *
     * @param autoWidthMethod  The method of automatically calculating the candle width.
     *
     * @see org.jfree.chart.renderer.CandlestickRenderer#WIDTHMETHOD_AVERAGE
     * @see org.jfree.chart.renderer.CandlestickRenderer#WIDTHMETHOD_SMALLEST
     * @see org.jfree.chart.renderer.CandlestickRenderer#WIDTHMETHOD_INTERVALDATA
     * @see org.jfree.chart.renderer.CandlestickRenderer#setCandleWidth
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthGap
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthFactor
     * @see org.jfree.chart.renderer.CandlestickRenderer#setMaxCandleWidthInMilliseconds
     */
    public void setAutoWidthMethod(int autoWidthMethod) {
        if (this.autoWidthMethod != autoWidthMethod) {
            Integer oldValue = new Integer(this.autoWidthMethod);
            this.autoWidthMethod = autoWidthMethod;
            firePropertyChanged("CandlestickRenderer.autoWidthMethod", 
                                oldValue, new Integer(autoWidthMethod));
        }
    }

    /**
     * Returns the factor by which the available space automatically calculated for the candles 
     * will be multiplied to determine the actual width to use.
     *
     * @return The width factor (generally between 0.0 and 1.0).
     */
    public double getAutoWidthFactor() {
        return autoWidthFactor;
    }

    /**
     * Sets the factor by which the available space automatically calculated for the candles will 
     * be multiplied to determine the actual width to use.
     *
     * @param autoWidthFactor The width factor (generally between 0.0 and 1.0).
     * @see org.jfree.chart.renderer.CandlestickRenderer#setCandleWidth
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthMethod
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthGap
     * @see org.jfree.chart.renderer.CandlestickRenderer#setMaxCandleWidthInMilliseconds
     */
    public void setAutoWidthFactor(double autoWidthFactor) {
        if (this.autoWidthFactor != autoWidthFactor) {
            Double oldValue = new Double(this.autoWidthFactor);
            this.autoWidthFactor = autoWidthFactor;
            firePropertyChanged("CandlestickRenderer.autoWidthFactor", 
                                oldValue, new Double(autoWidthFactor));
        }
    }

    /**
     * Returns the amount of space to leave on the left and right of each candle when 
     * automatically calculating widths.
     *
     * @return The gap.
     */
    public double getAutoWidthGap() {
        return autoWidthGap;
    }

    /**
     * Sets the amount of space to leave on the left and right of each candle when automatically 
     * calculating widths.
     *
     * @param autoWidthGap The gap.
     * @see org.jfree.chart.renderer.CandlestickRenderer#setCandleWidth
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthMethod
     * @see org.jfree.chart.renderer.CandlestickRenderer#setAutoWidthFactor
     * @see org.jfree.chart.renderer.CandlestickRenderer#setMaxCandleWidthInMilliseconds
     */
    public void setAutoWidthGap(double autoWidthGap) {
        if (this.autoWidthGap != autoWidthGap) {
            Double oldValue = new Double(this.autoWidthGap);
            this.autoWidthGap = autoWidthGap;
            firePropertyChanged("CandlestickRenderer.autoWidthGap", 
                                oldValue, new Double(autoWidthGap));
        }
    }

    /**
     * Returns the paint used to fill candles when the price moves up from open
     * to close.
     *
     * @return The paint.
     */
    public Paint getUpPaint() {
        return this.upPaint;
    }

    /**
     * Sets the paint used to fill candles when the price moves up from open
     * to close.
     * <P>
     * Registered property change listeners are notified that the
     * "CandleStickRenderer.upPaint" property has changed.
     *
     * @param paint The paint.
     */
    public void setUpPaint(Paint paint) {

        Paint old = this.upPaint;
        this.upPaint = paint;
        firePropertyChanged("CandleStickRenderer.upPaint", old, paint);

    }

    /**
     * Returns the paint used to fill candles when the price moves down from
     * open to close.
     *
     * @return The paint.
     */
    public Paint getDownPaint() {
        return this.downPaint;
    }

    /**
     * Sets the paint used to fill candles when the price moves down from open
     * to close.
     * <P>
     * Registered property change listeners are notified that the
     * "CandleStickRenderer.downPaint" property has changed.
     *
     * @param paint  The paint.
     */
    public void setDownPaint(Paint paint) {
        Paint old = this.downPaint;
        this.downPaint = paint;
        firePropertyChanged("CandleStickRenderer.downPaint", old, paint);
    }

    /**
     * Returns a flag indicating whether or not volume bars are drawn on the
     * chart.
     *
     * @return <code>true</code> if volume bars are drawn on the chart.
     */
    public boolean drawVolume() {
        return this.drawVolume;
    }

    /**
     * Sets a flag that controls whether or not volume bars are drawn in the
     * background.
     *
     * @param flag The flag.
     */
    public void setDrawVolume(boolean flag) {

        if (this.drawVolume != flag) {
            this.drawVolume = flag;
            firePropertyChanged("CandlestickRenderer.drawVolume", null, new Boolean(flag));
        }

    }

    /**
     * Initialises the renderer then returns the number of 'passes' through the data that the
     * renderer will require (usually just one).  This method will be called before the first
     * item is rendered, giving the renderer an opportunity to initialise any
     * state information it wants to maintain.  The renderer can do nothing if it chooses.
     *
     * @param g2  the graphics device.
     * @param dataArea  the area inside the axes.
     * @param plot  the plot.
     * @param dataset  the data.
     * @param info  an optional info collection object to return data back to the caller.
     *
     * @return The number of passes the renderer requires.
     */
    public XYItemRendererState initialise(Graphics2D g2,
                                          Rectangle2D dataArea,
                                          XYPlot plot,
                                          XYDataset dataset,
                                          PlotRenderingInfo info) {
          
        // calculate the maximum allowed candle width from the axis...
        ValueAxis axis = plot.getDomainAxis();
        double x1 = axis.getLowerBound();
        double x2 = x1 + this.maxCandleWidthInMilliseconds;
        RectangleEdge edge = plot.getDomainAxisEdge();
        double xx1 = axis.translateValueToJava2D(x1, dataArea, edge);
        double xx2 = axis.translateValueToJava2D(x2, dataArea, edge);
        this.maxCandleWidth = Math.abs(xx2 - xx1); // Absolute value, since the relative x 
                                         // positions are reversed for horizontal orientation
        
        // calculate the highest volume in the dataset... 
        if (this.drawVolume) {
            HighLowDataset highLowDataset = (HighLowDataset) dataset;
            this.maxVolume = 0.0;
            for (int series = 0; series < highLowDataset.getSeriesCount(); series++) {
                for (int item = 0; item < highLowDataset.getItemCount(series); item++) {
                    double volume = highLowDataset.getVolumeValue(series, item).doubleValue();
                    if (volume > this.maxVolume) {
                        this.maxVolume = volume;
                    }
                    
                }    
            }
        }
        
        return new XYItemRendererState(info);
    }

    /**
     * Draws the visual representation of a single data item.
     *
     * @param g2  the graphics device.
     * @param state  the renderer state.
     * @param dataArea  the area within which the plot is being drawn.
     * @param info  collects info about the drawing.
     * @param plot  the plot (can be used to obtain standard color information etc).
     * @param domainAxis  the domain axis.
     * @param rangeAxis  the range axis.
     * @param dataset  the dataset.
     * @param series  the series index (zero-based).
     * @param item  the item index (zero-based).
     * @param crosshairInfo  information about crosshairs on a plot.
     * @param pass  the pass index.
     */
    public void drawItem(Graphics2D g2, 
                         XYItemRendererState state,
                         Rectangle2D dataArea,
                         PlotRenderingInfo info,
                         XYPlot plot, 
                         ValueAxis domainAxis, 
                         ValueAxis rangeAxis,
                         XYDataset dataset, 
                         int series, 
                         int item,
                         CrosshairInfo crosshairInfo,
                         int pass) {

        boolean horiz;
        PlotOrientation orientation = plot.getOrientation();
        if (orientation == PlotOrientation.HORIZONTAL) {
            horiz = true;
        }
        else if (orientation == PlotOrientation.VERTICAL) {
            horiz = false;
        }
        else {
            return;
        }
        
        // setup for collecting optional entity info...
        EntityCollection entities = null;
        if (info != null) {
            entities = info.getOwner().getEntityCollection();
        }

        HighLowDataset highLowData = (HighLowDataset) dataset;

        Number x = highLowData.getXValue(series, item);
        Number yHigh = highLowData.getHighValue(series, item);
        Number yLow = highLowData.getLowValue(series, item);
        Number yOpen = highLowData.getOpenValue(series, item);
        Number yClose = highLowData.getCloseValue(series, item);

        RectangleEdge domainEdge = plot.getDomainAxisEdge();
        double xx = domainAxis.translateValueToJava2D(x.doubleValue(), dataArea, domainEdge);

        RectangleEdge edge = plot.getRangeAxisEdge();
        double yyHigh = rangeAxis.translateValueToJava2D(yHigh.doubleValue(), dataArea, edge);
        double yyLow = rangeAxis.translateValueToJava2D(yLow.doubleValue(), dataArea, edge);
        double yyOpen = rangeAxis.translateValueToJava2D(yOpen.doubleValue(), dataArea, edge);
        double yyClose = rangeAxis.translateValueToJava2D(yClose.doubleValue(), dataArea, edge);

        double volumeWidth;
        double stickWidth;
        if (candleWidth > 0) {
            // These are deliberately not bounded to minimums/maxCandleWidth to retain old 
            // behaviour.
            volumeWidth = candleWidth;
            stickWidth = candleWidth;
        }
        else {
            double xxWidth = 0;
            int itemCount;
            switch (autoWidthMethod) {
            
                case WIDTHMETHOD_AVERAGE:
                    itemCount = highLowData.getItemCount(series);
                    if (horiz) {
                        xxWidth = dataArea.getHeight() / itemCount;
                    }
                    else {
                        xxWidth = dataArea.getWidth() / itemCount;
                    }
                    break;
            
                case WIDTHMETHOD_SMALLEST:
                    // Note: It would be nice to pre-calculate this per series
                    itemCount = highLowData.getItemCount(series);
                    double lastPos = -1;
                    xxWidth = dataArea.getWidth();
                    for (int i = 0; i < itemCount; i++) {
                        double pos = domainAxis.translateValueToJava2D(
                            highLowData.getXValue(series, i).doubleValue(), dataArea, domainEdge);
                        if (lastPos != -1) {
                            xxWidth = Math.min(xxWidth, Math.abs(pos - lastPos));
                        }
                        lastPos = pos;
                    }
                    break;
            
                case WIDTHMETHOD_INTERVALDATA:
                    IntervalXYDataset intervalXYData = (IntervalXYDataset) dataset;
                    double startPos = domainAxis.translateValueToJava2D(
                        intervalXYData.getStartXValue(series, item).doubleValue(), dataArea, 
                        plot.getDomainAxisEdge());
                    double endPos = domainAxis.translateValueToJava2D(
                        intervalXYData.getEndXValue(series, item).doubleValue(), dataArea, 
                        plot.getDomainAxisEdge());
                    xxWidth = Math.abs(endPos - startPos);
                    break;
                
            }
            xxWidth -= 2 * autoWidthGap;
            xxWidth *= autoWidthFactor;
            xxWidth = Math.min(xxWidth, maxCandleWidth);
            volumeWidth = Math.max(Math.min(1, maxCandleWidth), xxWidth);
            stickWidth = Math.max(Math.min(3, maxCandleWidth), xxWidth);
        }

        Paint p = getItemPaint(series, item);
        Stroke s = getItemStroke(series, item);

        g2.setStroke(s);

        if (drawVolume) {
            int volume = highLowData.getVolumeValue(series, item).intValue();
            double volumeHeight = volume / this.maxVolume;

            double min, max;
            if (horiz) {
                min = dataArea.getMinX();
                max = dataArea.getMaxX();
            }
            else {
                min = dataArea.getMinY();
                max = dataArea.getMaxY();
            }

            double zzVolume = volumeHeight * (max - min);

            g2.setPaint(Color.gray);
            Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.3f));

            if (horiz) {
                g2.fill(new Rectangle2D.Double(min,
                                               xx - volumeWidth / 2,
                                               zzVolume, volumeWidth));
            }
            else {
                g2.fill(new Rectangle2D.Double(xx - volumeWidth / 2,
                                               max - zzVolume, volumeWidth, zzVolume));
            }

            g2.setComposite(originalComposite);
        }

        g2.setPaint(p);

        double yyMaxOpenClose = Math.max(yyOpen, yyClose);
        double yyMinOpenClose = Math.min(yyOpen, yyClose);
        double maxOpenClose = Math.max(yOpen.doubleValue(), yClose.doubleValue());
        double minOpenClose = Math.min(yOpen.doubleValue(), yClose.doubleValue());

        // draw the upper shadow
        if (yHigh.doubleValue() > maxOpenClose) {
            if (horiz) {
                g2.draw(new Line2D.Double(yyHigh, xx, yyMaxOpenClose, xx));
            }
            else {
                g2.draw(new Line2D.Double(xx, yyHigh, xx, yyMaxOpenClose));
            }
        }

        // draw the lower shadow
        if (yLow.doubleValue() < minOpenClose) {
            if (horiz) {
                g2.draw(new Line2D.Double(yyLow, xx, yyMinOpenClose, xx));
            }
            else {
                g2.draw(new Line2D.Double(xx, yyLow, xx, yyMinOpenClose));
            }
        }

        // draw the body
        Shape body = null;
        if (horiz) {
            body = new Rectangle2D.Double(yyMinOpenClose, xx - stickWidth / 2, 
                                          yyMaxOpenClose - yyMinOpenClose, stickWidth);
        } 
        else {
            body = new Rectangle2D.Double(xx - stickWidth / 2, yyMinOpenClose,
                                          stickWidth, yyMaxOpenClose - yyMinOpenClose);
        }
        if (yClose.doubleValue() > yOpen.doubleValue()) {
            if (upPaint != null) {
                g2.setPaint(upPaint);
                g2.fill(body);
            }
        }
        else {
            if (downPaint != null) {
                g2.setPaint(downPaint);
            }
            g2.fill(body);
        }
        g2.setPaint(p);
        g2.draw(body);

        // add an entity for the item...
        if (entities != null) {
            String tip = null;
            if (getToolTipGenerator() != null) {
                tip = getToolTipGenerator().generateToolTip(dataset, series, item);
            }
            String url = null;
            if (getURLGenerator() != null) {
                url = getURLGenerator().generateURL(dataset, series, item);
            }
            XYItemEntity entity = new XYItemEntity(body, dataset, series, item, tip, url);
            entities.addEntity(entity);
        }

    }

    /**
     * Tests this renderer for equality with another object.
     *
     * @param obj  the object.
     *
     * @return <code>true</code> or <code>false</code>.
     */
    public boolean equals(Object obj) {

        if (obj == null) {
            return false;
        }

        if (obj == this) {
            return true;
        }

        if (obj instanceof CandlestickRenderer) {
            CandlestickRenderer renderer = (CandlestickRenderer) obj;
            boolean result = super.equals(obj);
            result = result && (this.candleWidth == renderer.getCandleWidth());
            result = result && (this.upPaint.equals(renderer.getUpPaint()));
            result = result && (this.downPaint.equals(renderer.getDownPaint()));
            result = result && (this.drawVolume == renderer.drawVolume);
            return result;
        }

        return false;

    }

    /**
     * Returns a clone of the renderer.
     * 
     * @return A clone.
     * 
     * @throws CloneNotSupportedException  if the renderer cannot be cloned.
     */
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    /**
     * Provides serialization support.
     *
     * @param stream  the output stream.
     *
     * @throws IOException  if there is an I/O error.
     */
    private void writeObject(ObjectOutputStream stream) throws IOException {
        stream.defaultWriteObject();
        SerialUtilities.writePaint(this.upPaint, stream);
        SerialUtilities.writePaint(this.downPaint, stream);
    }

    /**
     * Provides serialization support.
     *
     * @param stream  the input stream.
     *
     * @throws IOException  if there is an I/O error.
     * @throws ClassNotFoundException  if there is a classpath problem.
     */
    private void readObject(ObjectInputStream stream) throws IOException, ClassNotFoundException {
        stream.defaultReadObject();
        this.upPaint = SerialUtilities.readPaint(stream);
        this.downPaint = SerialUtilities.readPaint(stream);
    }
    
}
