/* ===========================================================
 * JFreeChart : a free chart library for the Java(tm) platform
 * ===========================================================
 *
 * (C) Copyright 2000-2004, by Object Refinery Limited and Contributors.
 *
 * Project Info:  http://www.jfree.org/jfreechart/index.html
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc. 
 * in the United States and other countries.]
 *
 * ---------------------------
 * TimeSeriesURLGenerator.java
 * ---------------------------
 * (C) Copyright 2002, 2003, by Richard Atkinson and Contributors.
 *
 * Original Author:  Richard Atkinson (richard_c_atkinson@ntlworld.com);
 * Contributors:     David Gilbert (for Object Refinery Limited);
 *
 * $Id: TimeSeriesURLGenerator.java,v 1.3 2004/01/05 17:11:51 mungady Exp $
 *
 * Changes:
 * --------
 * 29-Aug-2002 : Initial version (RA);
 * 09-Oct-2002 : Fixed errors reported by Checkstyle (DG);
 * 23-Mar-2003 : Implemented Serializable (DG);
 *
 */
package org.jfree.chart.urls;

import java.io.Serializable;
import java.text.DateFormat;
import java.util.Date;

import org.jfree.data.XYDataset;

/**
 * A URL generator.
 *
 * @author Richard Atkinson
 */
public class TimeSeriesURLGenerator implements XYURLGenerator, Serializable {

    /** A formatter for the date. */
    private DateFormat dateFormat = DateFormat.getInstance();

    /** Prefix to the URL */
    private String prefix = "index.html";

    /** Name to use to identify the series */
    private String seriesParameterName = "series";

    /** Name to use to identify the item */
    private String itemParameterName = "item";

    /**
     * Blank constructor
     */
    public TimeSeriesURLGenerator() {
    }

    /**
     * Construct TimeSeriesURLGenerator overriding defaults
     *
     * @param dDateFormat  a formatter for the date.
     * @param sPrefix  the prefix of the URL.
     * @param sSeriesParameterName  the name of the series parameter in the URL.
     * @param sItemParameterName  the name of the item parameter in the URL.
     */
    public TimeSeriesURLGenerator(DateFormat dDateFormat, String sPrefix,
                                  String sSeriesParameterName, String sItemParameterName) {

        this.dateFormat = dDateFormat;
        this.prefix = sPrefix;
        this.seriesParameterName = sSeriesParameterName;
        this.itemParameterName = sItemParameterName;

    }

    /**
     * Generates a URL for a particular item within a series.
     *
     * @param dataset  the dataset.
     * @param series  the series number (zero-based index).
     * @param item  the item number (zero-based index).
     *
     * @return  the generated URL.
     */
    public String generateURL(XYDataset dataset, int series, int item) {
        String result = this.prefix;
        boolean firstParameter = result.indexOf("?") == -1;
        String seriesName = dataset.getSeriesName(series);
        if (seriesName != null) {
            result += firstParameter ? "?" : "&";
            result += this.seriesParameterName + "=" + seriesName;
            firstParameter = false;
        }

        long x = dataset.getXValue(series, item).longValue();
        String xValue = this.dateFormat.format(new Date(x));
        result += firstParameter ? "?" : "&";
        result += this.itemParameterName + "=" + xValue;

        return result;
    }


}
