/*
 * (c) Copyright: Artenum SARL, 101-103 Boulevard Mac Donald,
 *                75019, Paris, France 2005.
 *                http://www.artenum.com
 *
 * License:
 *
 *  This program is free software; you can redistribute it
 *  and/or modify it under the terms of the Q Public License;
 *  either version 1 of the License.
 *
 *  This program is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE. See the Q Public License for more details.
 *
 *  You should have received a copy of the Q Public License
 *  License along with this program;
 *  if not, write to:
 *    Artenum SARL, 101-103 Boulevard Mac Donald,
 *    75019, PARIS, FRANCE, e-mail: contact@artenum.com
 */
package com.artenum.cassandra.action;

import java.io.File;
import java.util.ArrayList;

import com.artenum.cassandra.pipeline.PipeLineManager;
import com.artenum.cassandra.ui.dialog.ThreadedProgressBarDialog;

/**
 * Threaded files loader. This class reads into a separated thread a VTK file or a list of 
 * files and load them into the pipeline manager. A progress monitor, like a progress bar
 * can be linked to this loader. 
 * 
 * Typical usage: 
 * 
 * <code>
 * ThreadeFileLoader loader = new ThreadeFileLoader(pipeLineManager);
 * loader.setProgressMonitor(new ThreadedProgressBarDialog());
 * for (int i = 0; i < vtkFiles.length; i++) {
 * loader.addFileToLoad(vtkFiles[i]);
 * }
 * loader.start();
 * </code>
 * 
 * or 
 * <code>
 * ThreadeFileLoader loader = new ThreadeFileLoader(pipeLineManager, vtkfile);
 * loader.setProgressMonitor(new ThreadedProgressBarDialog());
 * loader.start();
 * </code>
 * 
 * This loader performs also the validateViewAndGo action on the pipeline. 
 * 
 * @author Julien Forest, Artenum SARL
 *
 */
public class ThreadedFileLoader extends Thread{
    
    File file;
    ArrayList<File> fileList; 
    PipeLineManager pipeLineManager;
    ThreadedProgressBarDialog progressBarDialog;
    
    /**
     * Main contructor. 
     * 
     * @param pipeLineManager
     * @param file
     */
    public ThreadedFileLoader(PipeLineManager pipeLineManager, File file){
       fileList = new ArrayList<File>();
       fileList.add(file);
       this.pipeLineManager = pipeLineManager;
    }
    
    /**
     * Simple constructor. 
     * @param pipeLineManager
     */
    public ThreadedFileLoader(PipeLineManager pipeLineManager){
        fileList = new ArrayList<File>();
        this.pipeLineManager = pipeLineManager;
     }

    /**
     * Set a linked progress monitor. 
     * 
     * @param progressBarDialog
     */
    public void setProgressMonitor(ThreadedProgressBarDialog progressBarDialog){
        this.progressBarDialog = progressBarDialog;
        this.progressBarDialog.setProgressMinValue(0.0);
        this.progressBarDialog.setProgressMaxValue(50.0);
        this.progressBarDialog.showDialog(null, "");
    }
    
    /**
     * Set the list of files being loaded in case of multiple loading. 
     * 
     * @param fileList
     */
    public void setFileList(ArrayList<File> fileList){
        this.fileList = fileList;
    }
    
    /**
     * Add a file to load. 
     * @param file
     */
    public void addFileToLoad(File file){
        fileList.add(file);
    }
    
    
    public void run(){
        this.progressBarDialog.start();
        
        for (int index = 0; index < fileList.size(); index++){
        progressBarDialog.setMessage("Loading file, please wait...");
        progressBarDialog.setProgressValue(0.0);
        pipeLineManager.addVtkFile(fileList.get(index));
        
        progressBarDialog.setMessage("Loading into the pipeline...");
        progressBarDialog.setProgressValue(25.0);
        pipeLineManager.validateViewAndGo();
        
        progressBarDialog.setMessage("Loading into the pipeline...");
        progressBarDialog.setProgressValue(50.0);
        }
        progressBarDialog.close();
    }
}
