/**
 * Copyright (c) Artenum SARL 2004-2005
 * @author Sebastien Jourdain
 *
 * All rights reserved. This software can
 * not be used or copy or diffused without
 * an explicit license of Artenum SARL, Paris-France
 */
package com.artenum.cassandra.pipeline;

import java.util.ArrayList;
import java.util.Hashtable;

import javax.swing.JLabel;

import com.artenum.graph.impl.DefaultCell;
import com.artenum.graph.listener.DragListener;

/**
 * This class define a implementation of the the Cassandra's VtkObject interface which is a generic overload of
 * vtkObject for a better adaptation to the Cassandra pipeline manager.
 * 
 * Such object simplify the registration of the object into the pipeline, its management and the pipeline connectivity.
 * 
 * @author Sebastien Jourdain, Artenum SARL
 * @author Julien Forest, Artenum SARL
 * 
 * @version 1.2
 */
public class VtkObjectAdapter extends DefaultCell implements CassandraObject {
    private Object vtkObject;
    private int type = -1;
    private Integer id;
    private Integer localId;
    private String name;
    private final Hashtable<String, Object> metadata;
    private boolean valide;
    private final ArrayList<Integer> inputConnectivity;
    private final ArrayList<Integer> outputConnectivity;

    /**
     * Build a VtkObjectAdapter, i.e an adapter to the Cassandra's pipeline object, for the given vtkObject. This also
     * instantiates the connectivity tables for this CassandraObject.
     * 
     * @param vtkObject
     * @param type
     * @param name
     * @param id
     */
    public VtkObjectAdapter(final Object vtkObject, final int type, final String name, final Integer id) {
        super(vtkObject, new JLabel(name));
        new DragListener(this, getUI(), getUI());
        this.vtkObject = vtkObject;
        this.type = type;
        this.name = name;
        this.id = id;
        this.metadata = new Hashtable<String, Object>();
        this.valide = true;
        this.inputConnectivity = new ArrayList<Integer>();
        this.outputConnectivity = new ArrayList<Integer>();
    }

    /**
     * return the Id of the current VtkObject. The Id is the reference in the pipeline topology.
     */
    @Override
    public Integer getId() {
        return this.id;
    }

    /**
     * Return the type (e.g data-set or filter) of the current object.
     */
    @Override
    public int getType() {
        return this.type;
    }

    @Override
    public Object getVtkObject() {
        return this.vtkObject;
    }

    public void setId(final Integer id) {
        this.id = id;
    }

    /**
     * Set the VTK type of the linked vtkObjec see interface.
     * 
     * @param i
     */
    public void setType(final int i) {
        this.type = i;
    }

    public void setVtkObject(final Object object) {
        this.vtkObject = object;
    }

    @Override
    public String toString() {
        return getName();
    }

    @Override
    public Hashtable<String, Object> getMetaData() {
        return this.metadata;
    }

    @Override
    public String getName() {
        return this.name;
    }

    /**
     * Set the name of the current VtkObject.
     */
    @Override
    public void setName(final String name) {
        this.name = name;
    }

    @Override
    public boolean isValide() {
        return this.valide;
    }

    @Override
    public void setValide(final boolean valide) {
        this.valide = valide;
    }

    @Override
    public Integer getLocalTypeId() {
        return this.localId;
    }

    public void setLocalTypeId(final int localId) {
        this.localId = new Integer(localId);
    }

    /**
     * Add the VtkObject of Id vtkObjectId in input connectivity.
     * 
     * @param vtkObjectId
     */
    @Override
    public void addInputConnectivtiy(final int vtkObjectId) {
        this.inputConnectivity.add(vtkObjectId);
    }

    /**
     * Add the VtkObject vtkObjectIn in the input connectivity list.
     * 
     * @param vtkObjectIn
     */
    @Override
    public void addInputConnectivtiy(final CassandraObject vtkObjectIn) {
        this.inputConnectivity.add(vtkObjectIn.getId());
    }

    /**
     * 
     * 
     * @param VtkObjectId
     * @return
     */
    @Override
    public ArrayList<Integer> getInputConnectivityList() {
        return (this.inputConnectivity);
    }

    /**
     * return true if the VtkObject in input is in the input connectivity list.
     */
    @Override
    public boolean isInInputConnectivityInputList(final CassandraObject vtkObjectIn) {
        return (this.inputConnectivity.contains(vtkObjectIn.getId()));
    }

    /**
     * return true if the VtkObject of Id vtkObjectId in input is in the input connectivity list.
     */
    @Override
    public boolean isInInputConnectivityInputList(final int vtkObjectId) {
        return (this.inputConnectivity.contains(vtkObjectId));
    }

    /**
     * Add the VtkObject of Id vtkObjectId in output connectivity.
     * 
     * @param vtkObjectId
     */
    @Override
    public void addOutputConnectivity(final int vtkObjectId) {
        this.outputConnectivity.add(vtkObjectId);
    }

    /**
     * Add the VtkObject vtkObjectIn in the output connectivity list.
     * 
     * @param vtkObjectIn
     */
    @Override
    public void addOutputConnectivity(final CassandraObject vtkObjectIn) {
        this.outputConnectivity.add(vtkObjectIn.getId());
    }

    /**
     * @return return the output connectivity list.
     */
    @Override
    public ArrayList<Integer> getOutputConnectivityList() {
        return (this.outputConnectivity);
    }

    /**
     * return true if the VtkObject in input is in the input connectivity list.
     */
    @Override
    public boolean inOutputConnectivityList(final CassandraObject vtkObjectIn) {
        return (this.outputConnectivity.contains(vtkObjectIn.getId()));
    }

    /**
     * return true if the VtkObject of Id vtkObjectId in input is in the input connectivity list.
     */
    @Override
    public boolean inInputConnectivityList(final int vtkObjectId) {
        return (this.outputConnectivity.contains(vtkObjectId));
    }

    @Override
    public void clearInputConnectivityList() {
        this.inputConnectivity.clear();
    }

    @Override
    public void clearOutputConnecvityLsit() {
        this.outputConnectivity.clear();
    }

}
