package com.artenum.cassandra.util;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.JMenu;

public class RecentFilesManager extends JMenu implements ActionListener {

	public final static String PROPERTIES_PREFIX = "com.artenum.cassandra.recent.files";
	public final String LENGTH_FLAG = "length";
	public static final String RECENT_SELECTED_FILE_ACTION = "recent.selected.file.action";

	private ArrayList<File> recentFileList;
	private ArrayList<ActionListener> externalActionListener;
	private int nbMaxRecentFiles = 4;
	private boolean autoSave = true;
	private FileItem selectedItem;

	/**
	 * Main constructor.
	 */
	public RecentFilesManager() {
		recentFileList = new ArrayList<File>();
		externalActionListener = new ArrayList<ActionListener>();

		this.setText("Recent files");
	}

	public void addActionListener(ActionListener listener) {
		super.addActionListener(listener);
		externalActionListener.add(listener);
		for (int i = 0; i < this.getItemCount(); i++) {
			if (this.getItem(i) != null) {
				this.getItem(i).addActionListener(listener);
			}
		}
	}

	/**
	 * Add a file to the list of recent files in the limit of nbMaxRecentFiles.
	 * 
	 * @param file
	 */
	public void addFile(File file) {

		if (recentFileList.size() >= nbMaxRecentFiles) {
			recentFileList.remove(0);
		}
		recentFileList.add(file);

		updateView(file);
		if (autoSave) {
			saveFileListToProperties();
		}
	}

	private void updateView(File file) {
		FileItem item = new FileItem();
		item.setFile(file);
		item.setText(file.getAbsolutePath());
		item.setActionCommand(RECENT_SELECTED_FILE_ACTION);
		item.addActionListener(this);
		Iterator<ActionListener> iter = externalActionListener.iterator();
		while (iter.hasNext()) {
			item.addActionListener(iter.next());
		}
		this.add(item);
	}

	/**
	 * Save the list of files into the Java Properties. By default, this method
	 * is automatically called when the method addReference is called. To avoid
	 * this, set before the autoSave flag to false.
	 */
	public void saveFileListToProperties() {
		int listSize = 0;
		File file;

		if (recentFileList.size() < nbMaxRecentFiles) {
			listSize = recentFileList.size();
		} else {
			listSize = nbMaxRecentFiles;
		}

		// save the size of the current bookmark
		java.util.prefs.Preferences.userRoot().putInt(PROPERTIES_PREFIX + "." + LENGTH_FLAG, listSize);

		// save the bookmark
		Iterator<File> iter = this.recentFileList.iterator();
		while (iter.hasNext()) {
			file = iter.next();
			java.util.prefs.Preferences.userRoot().put(PROPERTIES_PREFIX + "." + this.recentFileList.indexOf(file), file.getAbsolutePath());
		}
	}

	/**
	 * read the file list from the java properties.
	 */
	public void readFileListFromProperties() {
		// read the size of the previously saved bookmark
		int bookMarkTmpSize = java.util.prefs.Preferences.userRoot().getInt(PROPERTIES_PREFIX + "." + LENGTH_FLAG, 0);
		// read the saved files/paths
		for (int i = 0; i < bookMarkTmpSize; i++) {
			addFile(new File(java.util.prefs.Preferences.userRoot().get(PROPERTIES_PREFIX + "." + i, new File(".").getAbsolutePath())));
		}
	}

	@Override
	public void actionPerformed(ActionEvent evt) {

		//File selectedFile;
		//String command = evt.getActionCommand();
		
		selectedItem = (FileItem) evt.getSource();
	}

	/**
	 * return the status of the auto save flag.
	 * 
	 * @return
	 */
	public boolean isAutoSave() {
		return autoSave;
	}

	/**
	 * Set the autoSave flag. If autoSave is true, the list of bookmarked files
	 * will be save to the Java properties automatically when the addReference
	 * method is called and a new path/file added.
	 * 
	 * @param autoSave
	 */
	public void setAutoSave(boolean autoSave) {
		this.autoSave = autoSave;
	}
	
	public FileItem getSelectedItem() {
		return selectedItem;
	}


	public void setSelectedItem(FileItem selectedItem) {
		this.selectedItem = selectedItem;
	}
	
}
