/*
 * (c) Copyright: Artenum SARL, 101-103 Boulevard Mac Donald,
 *                75019, Paris, France 2005.
 *                http://www.artenum.com
 *
 * License:
 *
 *  This program is free software; you can redistribute it
 *  and/or modify it under the terms of the Q Public License;
 *  either version 1 of the License.
 *
 *  This program is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE. See the Q Public License for more details.
 *
 *  You should have received a copy of the Q Public License
 *  License along with this program;
 *  if not, write to:
 *    Artenum SARL, 101-103 Boulevard Mac Donald,
 *    75019, PARIS, FRANCE, e-mail: contact@artenum.com
 */
package com.artenum.cassandra.plugin.cellcenter;

import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import vtk.vtkActor;
import vtk.vtkCellCenters;
import vtk.vtkDataSet;
import vtk.vtkLookupTable;
import vtk.vtkPolyDataMapper;
import vtk.vtkScalarBarActor;

import com.artenum.cassandra.pipeline.Filter;
import com.artenum.cassandra.pipeline.FilterImpl;
import com.artenum.cassandra.pipeline.PipeLineManager;
import com.artenum.cassandra.pipeline.RemoveListener;
import com.artenum.cassandra.pipeline.CassandraObject;
import com.artenum.cassandra.plugin.CassandraPlugInProperty;
import com.artenum.cassandra.plugin.CassandraPlugin;
import com.artenum.cassandra.plugin.PlugInControlUI;
import com.artenum.cassandra.plugin.PluginManager;

/**
 * <pre>
 * <b>Project ref           :</b> CASSANDRA project
 * <b>Copyright and license :</b> See relevant sections
 * <b>Status                :</b> under development
 * <b>Creation              :</b> 24/06/2010
 * <b>Modification          :</b>
 *
 * <b>Description  :</b>
 *             This class defines the VTK processing pipeline itself
 *             and the initialisation of the contextuel GUI of the plugin.
 *
 * @author        Julien Forest, ARTENUM SARL
 * 
 * @version       0.1
 */
public class CellCenterPlugin implements CassandraPlugin, RemoveListener {
    // Cassandra elements
    private String name;
    private PipeLineManager pipelineManager;
    private PluginManager pluginManager;
    private JPopupMenu contextualMenu;
    //private int nbUpdate = 0;

    // Contextual control GUI of the plugin
    private CellCenterControlUI controlUI;

    // Vtk data
    private vtkDataSet lastDataSet;
    private vtkLookupTable lookupTable;
    private vtkScalarBarActor scalarBar;
    private vtkCellCenters cellCenter;
    private vtkPolyDataMapper centerMapper;
    private vtkActor centerActor;
    private Filter filterImpl;

    // Pipeline VtkObjects
    private CassandraObject actor;
    private CassandraObject mapper;
    private CassandraObject dataset;
    private CassandraObject filter;

    /**
     * Main constructor and VTK pipeline definition.
     * @param pipelineManager is the central pipeline manager of Cassandra.
     * @param pluginManager is the central list of plugin instance, where a contextual menu access is given.
     */
    public CellCenterPlugin(PipeLineManager pipelineManager, PluginManager pluginManager, Frame owner) {
        // Init default variable
        this.name = "Cutting plane";
        this.pipelineManager = pipelineManager;
        this.pluginManager = pluginManager;
        this.filterImpl = new FilterImpl();
        filterImpl.addRemoveListener(this);

        // Init all the control UI
        controlUI = new CellCenterControlUI(pipelineManager, this, owner);

        // Init contextual menu 
        contextualMenu = new JPopupMenu("Cell Center filter menu");
        JMenuItem showControl = new JMenuItem("Show control");
        showControl.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    controlUI.setVisible(true);
                }
            });
        contextualMenu.add(showControl);
        contextualMenu.addSeparator();
        JMenuItem remove = new JMenuItem("Remove");
        remove.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    remove();
                }
            });
        contextualMenu.add(remove);

        // Define and initialise the VTK pipeline
        cellCenter = new vtkCellCenters();

        centerMapper = new vtkPolyDataMapper();
        centerMapper.SetInput(cellCenter.GetOutput());

        //we recover the common lookup table
        /*
           LookupTable Table = new LookupTable();
           Table.Build();
           cutMapper.SetLookupTable(Table.GetLookupTable());
           System.out.println("Color map =" + cutMapper.GetLookupTable().Print());
         */
        centerActor = new vtkActor();
        centerActor.SetMapper(centerMapper);

        // Build lookupTable
        lookupTable = new vtkLookupTable();
        lookupTable.SetHueRange(0.66667, 0);

        lookupTable.Build();
        centerMapper.SetLookupTable(lookupTable);

        // Scalar bar 
        scalarBar = new vtkScalarBarActor();
        scalarBar.SetLookupTable(lookupTable);

        // Load the result data in the Cassandra pipeline
        actor = pipelineManager.addActor(centerActor, "Cell Centers");
        mapper = pipelineManager.addMapper(centerMapper, "Cell Centers");
        dataset = pipelineManager.addDataSet(cellCenter.GetOutput(), "Cell Centers");
        filter = pipelineManager.addFilter(filterImpl, "Cell Centers");
        pipelineManager.addLookupTable(lookupTable, "Cell Centers");
        pipelineManager.setActorVisible(pipelineManager.addScalarBar(scalarBar, "Cell Centers"), true);

        //
        filter.getMetaData().put(CassandraObject.POPUP_MENU, getContextualMenu());

        // end of the VTK pipeline
        actor.setValide(false);
    }

    /**
     * Update the data in the defined pipeline
     * 
     */
    public void updateCellCenter(int vertexCellsFlag, vtkDataSet vtkDataSet) {
        if (vtkDataSet == null) {
            return;
        }

        actor.setValide(true);

        //
        cellCenter.SetInput(vtkDataSet);
        cellCenter.SetVertexCells(vertexCellsFlag);
        
        if (vtkDataSet.GetPointData().GetScalars() != null || vtkDataSet.GetCellData().GetScalars() != null) {
            if (((lastDataSet == null) || !lastDataSet.equals(vtkDataSet)) && centerMapper.GetLookupTable().equals(lookupTable)) {
                centerMapper.SetScalarRange(vtkDataSet.GetScalarRange());
                lookupTable.SetTableRange(vtkDataSet.GetScalarRange());
                lastDataSet = vtkDataSet;
            }
        }
        
        // Set the default actor visible by default
        //if ((nbUpdate++ == 0) && pipelineManager.getActorList().getData().contains(actor)) {
        if (pipelineManager.getActorList().getData().contains(actor)) {
            pipelineManager.setActorVisible(actor, true);
        }
        // and we hide the actor corresponding to the source
        pipelineManager.hideInputActor(actor, vtkDataSet);

        // Graph
        filterImpl.getInputDataSet().clear();
        filterImpl.getInputDataSet().add(vtkDataSet);
        filterImpl.getOutputDataSet().clear();
        filterImpl.getOutputDataSet().add(cellCenter.GetOutput());
        pipelineManager.notifyConnectivityChange(filter);
    }

    /**
     * Default plugin interface
     */
    public String getName() {
        return name;
    }

    public JPopupMenu getContextualMenu() {
        return contextualMenu;
    }

    public void remove() {
        pluginManager.removePlugin(this);
        pipelineManager.removeVtkObject(actor);
        pipelineManager.removeVtkObject(mapper);
        pipelineManager.removeVtkObject(dataset);
        pipelineManager.removeVtkObject(filter);
    }

    /**
     * String printed in the plugin manager list
     */
    public String toString() {
        return getName();
    }
    
    public Filter getInternalFilter() {
        return filterImpl;
    }

    @Override
    public CassandraPlugInProperty getPlugInProperty() {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public PlugInControlUI getControlUI() {
        return controlUI;
    }

    @Override
    public void initAndUpdate(CassandraPlugInProperty prop) {
        // TODO Auto-generated method stub
        
    }
}
