/*
 * (c) Copyright: Artenum SARL, 101-103 Boulevard Mac Donald,
 *                75019, Paris, France 2005.
 *                http://www.artenum.com
 *
 * License:
 *
 *  This program is free software; you can redistribute it
 *  and/or modify it under the terms of the Q Public License;
 *  either version 1 of the License.
 *
 *  This program is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE. See the Q Public License for more details.
 *
 *  You should have received a copy of the Q Public License
 *  License along with this program;
 *  if not, write to:
 *    Artenum SARL, 101-103 Boulevard Mac Donald,
 *    75019, PARIS, FRANCE, e-mail: contact@artenum.com
 */
package com.artenum.cassandra.plugin.cutting;

import com.artenum.cassandra.pipeline.Filter;
import com.artenum.cassandra.pipeline.FilterImpl;
import com.artenum.cassandra.pipeline.PipeLineManager;
import com.artenum.cassandra.pipeline.RemoveListener;
import com.artenum.cassandra.pipeline.VtkObject;
import com.artenum.cassandra.plugin.CassandraPlugin;
import com.artenum.cassandra.plugin.PluginManager;

import vtk.vtkActor;
import vtk.vtkCutter;
import vtk.vtkDataSet;
import vtk.vtkLookupTable;
import vtk.vtkPlane;
import vtk.vtkPolyDataMapper;
import vtk.vtkScalarBarActor;

import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

/**
 * <pre>
 * <b>Project ref           :</b> CASSANDRA project
 * <b>Copyright and license :</b> See relevant sections
 * <b>Status                :</b> under development
 * <b>Creation              :</b> 04/03/2005
 * <b>Modification          :</b>
 *
 * <b>Description  :</b>
 *             This class defines the VTK processing pipeline itself
 *             and the initialisation of the contextuel GUI of the plugin.
 *
 * </pre>
 * <table cellpadding="3" cellspacing="0" border="1" width="100%">
 * <tr BGCOLOR="#CCCCFF" CLASS="TableHeadingColor"><td><b>Version number</b></td><td><b>Author (name, e-mail)</b></td><td><b>Corrections/Modifications</b></td></tr>
 * <tr><td>0.1</td><td>Sebastien Jourdain, jourdain@artenum.com</td><td>Creation</td></tr>
 * </table>
 *
 * @author        Sebastien Jourdain
 * @version       0.1
 */
public class CuttingPlanePlugin implements CassandraPlugin, RemoveListener {
    // Cassandra elements
    private String name;
    private PipeLineManager pipelineManager;
    private PluginManager pluginManager;
    private JPopupMenu contextualMenu;
    private int nbUpdate = 0;

    // Contextual control GUI of the plugin
    private CuttingPlaneControlUI controlUI;

    // Vtk data
    private vtkDataSet lastDataSet;
    private vtkLookupTable lookupTable;
    private vtkScalarBarActor scalarBar;
    private vtkPlane plane;
    private vtkCutter planeCut;
    private vtkPolyDataMapper cutMapper;
    private vtkActor cutActor;
    private Filter filerImpl;

    // Pipeline VtkObjects
    private VtkObject actor;
    private VtkObject mapper;
    private VtkObject dataset;
    private VtkObject filter;

    /**
     * Main contructor and VTK pipeline definition.
     * @param pipelineManager is the central pipeline manager of Cassandra.
     * @param pluginManager is the central list of plugin instance, where a contextual menu access is given.
     */
    public CuttingPlanePlugin(PipeLineManager pipelineManager, PluginManager pluginManager, Frame owner) {
        // Init default variable
        this.name = "Cutting plane";
        this.pipelineManager = pipelineManager;
        this.pluginManager = pluginManager;
        this.filerImpl = new FilterImpl();
        filerImpl.addRemoveListener(this);

        // Init all the control UI
        controlUI = new CuttingPlaneControlUI(pipelineManager, this, owner);

        // Init contextual menu 
        contextualMenu = new JPopupMenu("Cutting plane menu");
        JMenuItem showControl = new JMenuItem("Show control");
        showControl.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    controlUI.setVisible(true);
                }
            });
        contextualMenu.add(showControl);
        contextualMenu.addSeparator();
        JMenuItem remove = new JMenuItem("Remove");
        remove.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    remove();
                }
            });
        contextualMenu.add(remove);

        // Define and init the VTK pipeline
        plane = new vtkPlane();
        plane.SetOrigin(0.0, 0.0, 0.0);
        plane.SetNormal(0, 0, 1.0);

        planeCut = new vtkCutter();
        planeCut.SetCutFunction(plane);

        cutMapper = new vtkPolyDataMapper();
        cutMapper.SetInput(planeCut.GetOutput());

        //we recover the common lookup table
        /*
           LookupTable Table = new LookupTable();
           Table.Build();
           cutMapper.SetLookupTable(Table.GetLookupTable());
           System.out.println("Color map =" + cutMapper.GetLookupTable().Print());
         */
        cutActor = new vtkActor();
        cutActor.SetMapper(cutMapper);

        // Build lookupTable
        lookupTable = new vtkLookupTable();
        lookupTable.SetHueRange(0.66667, 0);

        lookupTable.Build();
        cutMapper.SetLookupTable(lookupTable);

        // Scalar bar
        vtkScalarBarActor scalBar = new vtkScalarBarActor();
        scalBar.SetLookupTable(lookupTable);

        // Load the result data in the Cassandra pipeline
        actor = pipelineManager.addActor(cutActor, "Cutting plane");
        mapper = pipelineManager.addMapper(cutMapper, "Cutting plane");
        dataset = pipelineManager.addDataSet(planeCut.GetOutput(), "Cutting plane");
        filter = pipelineManager.addFilter(filerImpl, "Cutting plane");
        pipelineManager.addLookupTable(lookupTable, "Cutting plane");
        pipelineManager.setActorVisible(pipelineManager.addScalarBar(scalBar, "Cutting plane"), true);

        //
        filter.getMetaData().put(VtkObject.POPUP_MENU, getContextualMenu());

        // end of the VTK pipeline
        actor.setValide(false);
    }

    /**
     * Update the data in the defined pipeline
     * @param cx
     * @param cy
     * @param cz
     * @param nx
     * @param ny
     * @param nz
     * @param vtkDataSet
     */
    public void updateCuttingPlane(double cx, double cy, double cz, double nx, double ny, double nz, vtkDataSet vtkDataSet) {
        if (vtkDataSet == null) {
            return;
        }

        actor.setValide(true);

        //
        plane.SetOrigin(cx, cy, cz);
        plane.SetNormal(nx, ny, nz);
        planeCut.SetInput(vtkDataSet);
        if (vtkDataSet.GetPointData().GetScalars() != null) {
            if (((lastDataSet == null) || !lastDataSet.equals(vtkDataSet)) && cutMapper.GetLookupTable().equals(lookupTable)) {
                cutMapper.SetScalarRange(vtkDataSet.GetScalarRange());
                lookupTable.SetTableRange(vtkDataSet.GetScalarRange());
                lastDataSet = vtkDataSet;
            }
        }

        // Set the default actor visible by default
        if ((nbUpdate++ == 0) && pipelineManager.getActorList().getData().contains(actor)) {
            pipelineManager.setActorVisible(actor, true);
        }

        // Graph
        filerImpl.getInputDataSet().clear();
        filerImpl.getInputDataSet().add(vtkDataSet);
        filerImpl.getOutputDataSet().clear();
        filerImpl.getOutputDataSet().add(planeCut.GetOutput());
        pipelineManager.notifyConnectivityChange(filter);
    }

    /**
     * Default plugin interface
     */
    public String getName() {
        return name;
    }

    public JPopupMenu getContextualMenu() {
        return contextualMenu;
    }

    public void remove() {
        pluginManager.removePlugin(this);
        pipelineManager.removeVtkObject(actor);
        pipelineManager.removeVtkObject(mapper);
        pipelineManager.removeVtkObject(dataset);
        pipelineManager.removeVtkObject(filter);
    }

    /**
     * String printed in the plugin manager list
     */
    public String toString() {
        return getName();
    }
}
