/**
 * Copyright (c) Artenum SARL 2004-2005
 * @author Julien Forest / Sebastien Jourdain
 *
 * All rights reserved. This software can
 * not be used or copy or diffused without
 * an explicit license of Artenum SARL, Paris-France
 */
package com.artenum.cassandra.plugin.info2d;

import com.artenum.cassandra.pipeline.Filter;
import com.artenum.cassandra.pipeline.FilterImpl;
import com.artenum.cassandra.pipeline.PipeLineManager;
import com.artenum.cassandra.pipeline.RemoveListener;
import com.artenum.cassandra.pipeline.CassandraObject;
import com.artenum.cassandra.plugin.CassandraPlugInProperty;
import com.artenum.cassandra.plugin.CassandraPlugin;
import com.artenum.cassandra.plugin.PlugInControlUI;
import com.artenum.cassandra.plugin.PluginManager;

import vtk.vtkActor2D;
import vtk.vtkDataSet;
import vtk.vtkLabeledDataMapper;

import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

/**
 * <pre>
 * <b>Project ref           :</b> CASSANDRA project
 * <b>Copyright and license :</b> See relevant sections
 * <b>Status                :</b> under development
 * <b>Creation              :</b> 04/03/2005
 * <b>Modification          :</b>
 *
 * <b>Description  :</b>
 *             This class defines the VTK processing pipeline itself
 *             and the initialisation of the contextuel GUI of the plugin.
 *
 * </pre>
 * <table cellpadding="3" cellspacing="0" border="1" width="100%">
 * <tr BGCOLOR="#CCCCFF" CLASS="TableHeadingColor"><td><b>Version number</b></td><td><b>Author (name, e-mail)</b></td><td><b>Corrections/Modifications</b></td></tr>
 * <tr><td>0.1</td><td>Sebastien Jourdain, jourdain@artenum.com</td><td>Creation</td></tr>
 * </table>
 *
 * @author        Sebastien Jourdain ARTENUM SARL
 * @author        Julien Forest, ARTENUM SARL
 * 
 * @version       1.2
 */
public class Infos2DPlugin implements CassandraPlugin, RemoveListener {
    public final static int SHOW_ID = 0;
    public final static int SHOW_DATA_FIELD = 1;
    public final static int SHOW_NORMAL = 2;
    public final static int SHOW_SCALAR = 3;
    public final static int SHOW_TCOORD = 4;
    public final static int SHOW_TENSOR = 5;
    public final static int SHOW_VECTOR = 6;

    //
    private String name;
    private PipeLineManager pipelineManager;
    private PluginManager pluginManager;
    private JPopupMenu contextualMenu;
    private Info2DControlUI controlUI;
    private int nbUpdate = 0;

    // Vtk data
    private vtkDataSet datasetVTK;
    private vtkLabeledDataMapper mapperVTK;
    private vtkActor2D actorVTK;

    // Pipeline VtkObjects
    private CassandraObject actor;
    private CassandraObject mapper;
    private CassandraObject filter;
    private Filter filterImpl;

    public Infos2DPlugin() {
        this.name = "Info2D";
        filterImpl = new FilterImpl();
    }

    public Infos2DPlugin(PipeLineManager pipelineManager, PluginManager pluginManager, Frame owner) {
        this();
        this.pipelineManager = pipelineManager;
        this.pluginManager = pluginManager;

        // Init control UI
        controlUI = new Info2DControlUI(pipelineManager, this, owner);

        // Init contextual menu
        contextualMenu = new JPopupMenu("2D Info menu");
        JMenuItem showControl = new JMenuItem("Show control");
        showControl.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    controlUI.setVisible(true);
                }
            });
        contextualMenu.add(showControl);
        contextualMenu.addSeparator();
        JMenuItem remove = new JMenuItem("Remove");
        remove.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    remove();
                }
            });
        contextualMenu.add(remove);

        // Init pipeline
        // Here write your processing pipeline
        mapperVTK = new vtkLabeledDataMapper();
        mapperVTK.SetLabelFormat("%g");
        actorVTK = new vtkActor2D();
        actorVTK.SetMapper(mapperVTK);

        //
        filterImpl.addRemoveListener(this);
        filterImpl.getOutputActor().add(actorVTK);

        //
        actor = pipelineManager.addActor(actorVTK, "Info 2D");
        filter = pipelineManager.addFilter(filterImpl, "Info 2D");
        filter.getMetaData().put(CassandraObject.POPUP_MENU, getContextualMenu());

        //
        actor.setValide(false);
    }

    public void updateInfo2D(vtkDataSet inputDataSet, int option) {
        if (inputDataSet != null) {
            filterImpl.getInputDataSet().clear();
            filterImpl.getInputDataSet().add(inputDataSet);

            // Update mapper
            if (((datasetVTK == null) || !datasetVTK.equals(inputDataSet))) {
                datasetVTK = inputDataSet;
                mapperVTK.SetInput(datasetVTK);
            }

            switch (option) {
            case SHOW_ID:
                mapperVTK.SetLabelModeToLabelIds();
                break;
            case SHOW_DATA_FIELD:
                mapperVTK.SetLabelModeToLabelFieldData();
                break;
            case SHOW_NORMAL:
                mapperVTK.SetLabelModeToLabelNormals();
                break;
            case SHOW_SCALAR:
                mapperVTK.SetLabelModeToLabelScalars();
                break;
            case SHOW_TCOORD:
                mapperVTK.SetLabelModeToLabelTCoords();
                break;
            case SHOW_TENSOR:
                mapperVTK.SetLabelModeToLabelTensors();
                break;
            case SHOW_VECTOR:
                mapperVTK.SetLabelModeToLabelVectors();
                break;
            }

            // Set the default actor visible by default
            if (mapperVTK.GetInput().equals(datasetVTK) && pipelineManager.getActorList().getData().contains(actor)) {
                actor.setValide(true);
                pipelineManager.setActorVisible(actor, true);
            }

            pipelineManager.notifyConnectivityChange(filter);
        }
    }

    public String getName() {
        return name;
    }

    public void remove() {
        pipelineManager.removeVtkObject(actor);
        pipelineManager.removeVtkObject(mapper);
        pipelineManager.removeVtkObject(filter);
        pluginManager.removePlugin(this);
        /*
           mapperVTK.RemoveAllInputs();
           System.out.println("D");
           pluginManager.removePlugin(this);
           System.out.println("E");
           pipelineManager.removeVtkObject(actor);
           System.out.println("F");
           pipelineManager.removeVtkObject(mapper);
           System.out.println("G");
           pipelineManager.removeVtkObject(dataset);
           System.out.println("H");
           pipelineManager.removeVtkObject(filter);
           System.out.println("I");
         */
    }

    public JPopupMenu getContextualMenu() {
        return contextualMenu;
    }

    public String toString() {
        return getName();
    }
    
    public Filter getInternalFilter() {
        return filterImpl;
    }

    public PlugInControlUI getControlUI() {
        return controlUI;
    }

    public CassandraPlugInProperty getPlugInProperty() {
        // TODO Auto-generated method stub
        return null;
    }


    @Override
    public void initAndUpdate(CassandraPlugInProperty prop) {
        // TODO Auto-generated method stub
        
    }
}
