/*
 * (c) Copyright: 24, rue Louis Blanc, 75010, 
 *                Paris, France, 2010.
 *                http://www.artenum.com
 *
 * License:
 *
 *  This program is free software; you can redistribute it
 *  and/or modify it under the terms of the Q Public License;
 *  either version 1 of the License.
 *
 *  This program is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE. See the Q Public License for more details.
 *
 *  You should have received a copy of the Q Public License
 *  License along with this program;
 *  if not, write to:
 *    24, rue Louis Blanc, 75010, Paris, 
 *    e-mail: contact@artenum.com
 *
 * Initial Author:
 *    - Julien Forest, ARTENUM SARL
 */
package com.artenum.cassandra.renderer.vtk;

/**
 * Generic and safe VTK native context loaded. This class tries to load the
 * native layer (JNI) of the VTK's libraries if it can find them in the
 * LD_LIBRARY_PATH for Linux/Unix systems, in DYLD_LIBRARY_PATH for Apple MacOSX
 * systems or in the PATH for MS/Windows systems.
 * 
 * The CLASSPATH should include all jars related to VTK and the JNI layers (e.g
 * vtk.jar).
 * 
 * Theses path should be defined/set BEFORE launch the JVM.
 * 
 * See the Cassandra's or the Keridwen's projects pages for further information
 * and/or advices.
 * 
 * The present context loader return true (boolean) if the context has been
 * successfully loaded otherwise it returns false.
 * 
 * @author Julien Forest, Artenum SARL, Paris, France
 * @version 1.0
 * 
 */
public class VtkContextLoader {

	private int verboseLevel = 0;

	public final static String errorMessage = "VTK native context loading error. \n \n"
			+ "Impossible to load the native layer (JNI) of the VTK's libraries! \n"
			+ "We could not find them in the LD_LIBRARY_PATH for Linux/Unix systems, \n"
			+ "in DYLD_LIBRARY_PATH for Apple MacOSX systems or in the PATH for \n " 
			+ "MS/Windows systems; or we could nt able to load the native layer. \n"
			+ "Please check this. \n"
			+ "The CLASSPATH should include all jars related to VTK and the JNI \n " 
			+ "layers (e.g vtk.jar). Theses paths should be defined/set BEFORE \n"
			+ "launching the JVM.\n"
			+ "See the Cassandra's or the Keridwen's projects pages for further \n" 
			+ "information, supports and/or advices: \n" 
			+ " http://www.artenum.com/cassandra\n"
			+ " http://www.artenum.com/keridwen";

	/**
	 * Default constructor.
	 */
	public VtkContextLoader() {

	}

	public VtkContextLoader(int verboseLevel) {
		this.verboseLevel = verboseLevel;
	}

	/**
	 * Try to load the JNI VTK context.
	 * 
	 * 
	 * @return true (boolean) if the context has been successfully loaded
	 *         otherwise it returns false.
	 */
	public boolean load() {
		// now we try to display the result as a VTK data set
		try {
			System.out.print("Try to load the VTK native layer... ");
			System.loadLibrary("vtkCommonJava");
			System.loadLibrary("vtkRenderingJava");
			System.loadLibrary("vtkImagingJava");
			System.loadLibrary("vtkIOJava");
			System.loadLibrary("vtkFilteringJava");
			System.loadLibrary("vtkGraphicsJava");
			System.loadLibrary("vtkHybridJava");
			try {
				System.loadLibrary("vtkParallelJava");
			} catch (Throwable errorPara) {
				System.out.println("No vtkParallelJava present... not critical");
			}

			System.out.println("VTK context loaded.");
			return (true);
		} catch (Throwable e) {
			if (verboseLevel > 0) {
				System.out.println("Error in loading VTK layer, no VTK found in the LD_LIBRARY_PATH.");
				System.out.println("java.library.path= " + System.getProperty("java.library.path"));
			}
			;
			if (verboseLevel > 1) {
				System.out.println(e.getMessage());
			}
			;
			if (verboseLevel > 2) {
				System.out.println(errorMessage);
			}
			;
			return (false);
		}
	}

}
