/*
 * Project: Cassandra
 * 
 * (c) Copyright: Artenum SARL, 24 rue Louis Blanc, 
 *                75010, Paris, France 2005.
 *                http://www.artenum.com
 *
 * License:
 *
 *  This program is free software; you can redistribute it 
 *  and/or modify it under the terms of the Q Public License; 
 *  either version 1 of the License.
 * 
 *  This program is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the Q Public License for more details.
 * 
 *  You should have received a copy of the Q Public License 
 *  License along with this program;
 *  if not, write to:
 *    Artenum SARL, 24 rue Louis Blanc,
 *    75010, PARIS, FRANCE, e-mail: contact@artenum.com
 */
package com.artenum.cassandra.plugin;

import java.util.HashMap;

/**
 * Basic implementation of the plug-in properties interfaces. Such list can store all type
 * of object using all type of object as key. It also provides the getSafely methods to 
 * return a safely casted object. 
 * 
 * 
 * @author Julien Forest, Artenum SARL
 *
 */
public class SimplePlugInProperty implements CassandraPlugInProperty {

    private String name;
    private HashMap propList;

    /**
     * main constructor
     */
    public SimplePlugInProperty() {
        propList = new HashMap();

    }

    /**
     * Add the object prop into the list with the key. 
     */
    @Override
    public void put(Object key, Object prop) {
        propList.put(key, prop);
    }

    /**
     * get the object for the given key. This returns the stored 
     * object without any cast. Useful for a fast and simple access 
     * (equivalent to the get method on hashmaps). 
     */
    @Override
    public Object get(Object key) {
        return propList.get(key);
    }

    /**
     * This first checks if the properties list contains the object and,
     * if yes, returns it without any cast, otherwise returns the default
     * value given with the defaultValue parameter. 
     */
    @Override
    public Object getSafely(Object key, Object defaultValue) {
       return getObjectSafely(key, defaultValue);
    }
    
    
    private Object getObjectSafely(Object key, Object defaultValue) {
        if (this.containsKey(key)) {
            return (this.get(key));
        } else {
            System.err.println("key " + key.toString() + " not found. Value set to default");
            return (defaultValue);
        }
    }
    
    /**
     * This first checks if the properties list contains the object and,
     * if yes, returns it cast with the type of the defaultValue parameter, 
     * otherwise returns the defaultValue  
     */
    @Override
    public double getSafely(Object key, double defaultValue) {
        try {
            return (Double) getObjectSafely(key, defaultValue);
        } catch (Exception e) {
            System.err.println("Property not understood or impossible to cast. Set to the default value");
            return (defaultValue);
        }
    }
    
    /**
     * This first checks if the properties list contains the object and,
     * if yes, returns it cast with the type of the defaultValue parameter, 
     * otherwise returns the defaultValue  
     */
    @Override
    public int getSafely(Object key, int defaultValue) {
        try {
            return (Integer) getObjectSafely(key, defaultValue);
        } catch (Exception e) {
            System.err.println("Property not understood or impossible to cast. Set to the default value");
            return (defaultValue);
        }
    }

    /**
     * This first checks if the properties list contains the object and,
     * if yes, returns it cast with the type of the defaultValue parameter, 
     * otherwise returns the defaultValue  
     */
    @Override
    public float getSafely(Object key, float defaultValue) {
        try {
            return (Float) getObjectSafely(key, defaultValue );
        } catch (Exception e) {
            System.err.println("Property not understood or impossible to cast. Set to the default value");
            return (defaultValue);
        }
    }
    

    /**
     * returns the hashmap of the propertyList. 
     */
    @Override
    public HashMap getPropertyMap() {
        return propList;
    }

    /**
     * checks if the PropertyList contains the given key. 
     */
    @Override
    public boolean containsKey(Object key) {
        return (propList.containsKey(key));
    }

}
